/*******************************************************************************
**3456789 123456789 123456789 123456789 123456789 123456789 123456789 123456789
**      10        20        30        40        50        60        70        80
**
** notify-osd
**
** test-sd-bubble.c - unit-tests for snap-decision bubble class
**
** Copyright 2012 Canonical Ltd.
**
** Authors:
**    Mirco "MacSlow" Mueller <mirco.mueller@canonical.com>
**
** This program is free software: you can redistribute it and/or modify it
** under the terms of the GNU General Public License version 3, as published
** by the Free Software Foundation.
**
** This program is distributed in the hope that it will be useful, but
** WITHOUT ANY WARRANTY; without even the implied warranties of
** MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
** PURPOSE.  See the GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License along
** with this program.  If not, see <http://www.gnu.org/licenses/>.
**
*******************************************************************************/

#include <glib.h>
#include <gdk/gdk.h>

#include "config.h"
#include "sd-bubble.h"
#include "settings.h"

gboolean  g_sd_bubble_timed_out = FALSE;
gboolean  g_sd_bubble_close     = FALSE;
Settings* g_settings            = NULL;

static SdBubble*
sd_bubble_setup ()
{
	SdBubble* sd_bubble  = NULL;
	gchar*    actions[8] = {"accept_id",  "Accept",
                            "decline_id", "Decline",
                            "no_id",      "No",
                            "nope_id",    "Nope"};
	g_settings = settings_new ();

	sd_bubble = sd_bubble_new (actions, 8, g_settings);
	sd_bubble_set_dbus_sender (sd_bubble, ":1.1234");
	sd_bubble_set_state (sd_bubble, SD_BUBBLE_STATE_DEFAULT);
	g_assert (sd_bubble != NULL);
	g_assert (IS_SD_BUBBLE (sd_bubble));

	return sd_bubble;
}

static void
sd_bubble_clean_up (SdBubble* sd_bubble)
{
	g_object_unref (sd_bubble);
	sd_bubble = NULL;
	settings_del (g_settings);
	g_settings = NULL;
}

static gboolean
stop_sd_main_loop (GMainLoop* loop)
{
	g_main_loop_quit (loop);

	return FALSE;
}

static void
test_sd_bubble_new ()
{
	SdBubble* sd_bubble = sd_bubble_setup ();
	sd_bubble_clean_up (sd_bubble);
}

static void
test_sd_bubble_getset_summary ()
{
	SdBubble* sd_bubble = sd_bubble_setup ();
	gchar*    sample1   = "Title";
	gchar*    sample2   = "Summary";

	sd_bubble_set_summary (sd_bubble, sample1);
	g_assert_cmpstr (sd_bubble_get_summary (sd_bubble), ==, sample1);
	sd_bubble_set_summary (sd_bubble, sample2);
	g_assert_cmpstr (sd_bubble_get_summary (sd_bubble), ==, sample2);

	sd_bubble_clean_up (sd_bubble);
}

static void
test_sd_bubble_getset_body ()
{
	SdBubble* sd_bubble = sd_bubble_setup ();
	gchar*    sample1   = "This is some odd example body-text.";
	gchar*    sample2   = "Just some other strange sample-text.";

	sd_bubble_set_body (sd_bubble, sample1);
	g_assert_cmpstr (sd_bubble_get_body (sd_bubble), ==, sample1);
	sd_bubble_set_body (sd_bubble, sample2);
	g_assert_cmpstr (sd_bubble_get_body (sd_bubble), ==, sample2);

	sd_bubble_clean_up (sd_bubble);
}

static void
test_sd_bubble_set_icon ()
{
	SdBubble*  sd_bubble = sd_bubble_setup ();
	GdkPixbuf* pixbuf    = NULL;

	/* create pixbuf for testing */
	pixbuf = gdk_pixbuf_new (GDK_COLORSPACE_RGB, TRUE, 8, 100, 100);
	g_assert (pixbuf);
	sd_bubble_set_icon_pixbuf (sd_bubble, pixbuf);

	g_object_unref (pixbuf);
	sd_bubble_clean_up (sd_bubble);
}

static void
test_sd_bubble_get_num_actions ()
{
	SdBubble* sd_bubble   = NULL;
	gchar*    actions[10] = {"accept_id",  "Accept",
							 "decline_id", "Decline",
							 "no_id",      "No",
							 "nein_id",    "Nein",
							 "non_id",     "Non"};
	guint     num_actions = 0;

	g_settings = settings_new ();
	sd_bubble = sd_bubble_new (actions, 10, g_settings);
	g_assert (sd_bubble != NULL);
	g_assert (IS_SD_BUBBLE(sd_bubble));

	num_actions = sd_bubble_get_num_actions (sd_bubble);
	g_assert_cmpuint (num_actions, ==, 10);

	g_object_unref (sd_bubble);
	settings_del (g_settings);
	g_settings = NULL;
}

static void
test_sd_bubble_getset_actions ()
{
	SdBubble* sd_bubble    = NULL;
	gchar*    actions5[10] = {"accept_id",  "Accept",
							  "decline_id", "Decline",
							  "no_id",      "No",
							  "nein_id",    "Nein",
							  "non_id",     "Non"};
	gchar*    actions4[8]  = {"ok_id",     "Ok",
							  "reject_id", "Reject",
							  "dont_id",   "Don't",
							  "stop_id",   "Stop"};
	guint     num_actions  = 0;

	g_settings = settings_new ();
	sd_bubble = sd_bubble_new (actions5, 10, g_settings);
	g_assert (sd_bubble != NULL);
	g_assert (IS_SD_BUBBLE(sd_bubble));

	num_actions = sd_bubble_get_num_actions (sd_bubble);
	g_assert_cmpuint (num_actions, ==, 10);

	g_assert_cmpstr (sd_bubble_get_action_by_index (sd_bubble, 0), ==, "accept_id");
	g_assert_cmpstr (sd_bubble_get_action_by_index (sd_bubble, 1), ==, "Accept");
	g_assert_cmpstr (sd_bubble_get_action_by_index (sd_bubble, 2), ==, "decline_id");
	g_assert_cmpstr (sd_bubble_get_action_by_index (sd_bubble, 3), ==, "Decline");
	g_assert_cmpstr (sd_bubble_get_action_by_index (sd_bubble, 4), ==, "no_id");
	g_assert_cmpstr (sd_bubble_get_action_by_index (sd_bubble, 5), ==, "No");
	g_assert_cmpstr (sd_bubble_get_action_by_index (sd_bubble, 6), ==, "nein_id");
	g_assert_cmpstr (sd_bubble_get_action_by_index (sd_bubble, 7), ==, "Nein");
	g_assert_cmpstr (sd_bubble_get_action_by_index (sd_bubble, 8), ==, "non_id");
	g_assert_cmpstr (sd_bubble_get_action_by_index (sd_bubble, 9), ==, "Non");

	sd_bubble_set_actions (sd_bubble, actions4, 8);
	num_actions = sd_bubble_get_num_actions (sd_bubble);
	g_assert_cmpuint (num_actions, ==, 8);

	g_assert_cmpstr (sd_bubble_get_action_by_index (sd_bubble, 0), ==, "ok_id");
	g_assert_cmpstr (sd_bubble_get_action_by_index (sd_bubble, 1), ==, "Ok");
	g_assert_cmpstr (sd_bubble_get_action_by_index (sd_bubble, 2), ==, "reject_id");
	g_assert_cmpstr (sd_bubble_get_action_by_index (sd_bubble, 3), ==, "Reject");
	g_assert_cmpstr (sd_bubble_get_action_by_index (sd_bubble, 4), ==, "dont_id");
	g_assert_cmpstr (sd_bubble_get_action_by_index (sd_bubble, 5), ==, "Don't");
	g_assert_cmpstr (sd_bubble_get_action_by_index (sd_bubble, 6), ==, "stop_id");
	g_assert_cmpstr (sd_bubble_get_action_by_index (sd_bubble, 7), ==, "Stop");

	g_object_unref (sd_bubble);
	settings_del (g_settings);
	g_settings = NULL;
}

static void
test_sd_bubble_get_x ()
{
	SdBubble* sd_bubble = sd_bubble_setup ();

	sd_bubble_move (sd_bubble, 20, 40);
	g_assert_cmpint (sd_bubble_get_x (sd_bubble), ==, 20);

	sd_bubble_clean_up (sd_bubble);
}

static void
test_sd_bubble_get_y ()
{
	SdBubble* sd_bubble = sd_bubble_setup ();

	sd_bubble_move (sd_bubble, 20, 40);
	g_assert_cmpint (sd_bubble_get_y (sd_bubble), ==, 40);

	sd_bubble_clean_up (sd_bubble);
}

static void
test_sd_bubble_get_width ()
{
	SdBubble* sd_bubble  = sd_bubble_setup ();

	g_assert_cmpint (sd_bubble_get_width (sd_bubble), >=, 0);

	sd_bubble_clean_up (sd_bubble);
}

static void
test_sd_bubble_get_height ()
{
	SdBubble* sd_bubble  = sd_bubble_setup ();

	g_assert_cmpint (sd_bubble_get_height (sd_bubble), >=, 0);

	sd_bubble_clean_up (sd_bubble);
}

static void
test_sd_bubble_getset_state ()
{
	SdBubble* sd_bubble  = sd_bubble_setup ();

	sd_bubble_set_state (sd_bubble, SD_BUBBLE_STATE_CONTRACTED);
	g_assert_cmpint (sd_bubble_get_state (sd_bubble),
	                 ==,
	                 SD_BUBBLE_STATE_CONTRACTED);
	sd_bubble_set_state (sd_bubble, SD_BUBBLE_STATE_DEFAULT);
	g_assert_cmpint (sd_bubble_get_state (sd_bubble),
	                 ==,
	                 SD_BUBBLE_STATE_DEFAULT);
	sd_bubble_set_state (sd_bubble, SD_BUBBLE_STATE_EXPANDED);
	g_assert_cmpint (sd_bubble_get_state (sd_bubble),
	                 ==,
	                 SD_BUBBLE_STATE_EXPANDED);

	sd_bubble_clean_up (sd_bubble);
}

static void
test_sd_bubble_getset_notification_id ()
{
	SdBubble* sd_bubble = sd_bubble_setup ();
	guint id1 = 12;
	guint id2 = 42;

	sd_bubble_set_notification_id (sd_bubble, id1);
	g_assert_cmpuint (sd_bubble_get_notification_id (sd_bubble), ==, id1);
	sd_bubble_set_notification_id (sd_bubble, id2);
	g_assert_cmpuint (sd_bubble_get_notification_id (sd_bubble), ==, id2);

	sd_bubble_clean_up (sd_bubble);
}

static void
test_sd_bubble_show ()
{
	SdBubble*  sd_bubble[3] = {NULL, NULL, NULL};
	gchar*     actions[8]   = {"accept_id",  "Accept",
                               "decline_id", "Decline",
                               "nope_id",    "Nope",
                               "never_id",   "Never"};
	GMainLoop* loop         = NULL;
	gint       i            = 0;

	g_settings = settings_new ();
	for (i = 0; i < 3; i++)
	{
		sd_bubble[i] = sd_bubble_new (actions, 8, g_settings);
		g_assert (sd_bubble[i] != NULL);
		g_assert (IS_SD_BUBBLE (sd_bubble[i]));
	}

	sd_bubble_set_state (sd_bubble[0], SD_BUBBLE_STATE_CONTRACTED);
	sd_bubble_set_state (sd_bubble[1], SD_BUBBLE_STATE_DEFAULT);
	sd_bubble_set_state (sd_bubble[2], SD_BUBBLE_STATE_EXPANDED);

	for (i = 0; i < 3; i++)
		sd_bubble_show (sd_bubble[i]);

	sd_bubble_move (sd_bubble[0], 500, 500);
	sd_bubble_move (sd_bubble[1],
	                500,
	                500 + sd_bubble_get_height (sd_bubble[0]) + 10);
	sd_bubble_move (sd_bubble[2],
	                500,
	                500 + sd_bubble_get_height (sd_bubble[0]) +
	                sd_bubble_get_height (sd_bubble[1]) + 20);

	/* let the main loop run */
	loop = g_main_loop_new (NULL, FALSE);
	g_timeout_add (1000, (GSourceFunc) stop_sd_main_loop, loop);
	g_main_loop_run (loop);

	for (i = 0; i < 3; i++)
		g_object_unref (sd_bubble[i]);

	settings_del (g_settings);
	g_settings = NULL;
}

/* FIXME: This needs a real working environment with a valid DBus-connection...
 * replicating all this just in the unit-test is too timeconsuming for the time
 * being... has to be added later
static gboolean
sd_bubble_close_cb (gpointer data)
{
	SdBubble* sd_bubble = SD_BUBBLE (data);

	sd_bubble_move (sd_bubble, 500, 500);
	sd_bubble_close (sd_bubble);
	g_sd_bubble_close = sd_bubble_is_visible (sd_bubble) == FALSE;
}

static void
test_sd_bubble_close ()
{
	SdBubble*  sd_bubble = sd_bubble_setup ();
	GMainLoop* loop      = NULL;

	sd_bubble_set_state (sd_bubble, SD_BUBBLE_STATE_DEFAULT);
	sd_bubble_show (sd_bubble);

	sd_bubble_move (sd_bubble, 500, 500);
	sd_bubble_close (sd_bubble);
	g_assert_cmpint (sd_bubble_is_visible (sd_bubble), ==, FALSE);

	g_timeout_add_seconds (1, sd_bubble_close_cb, sd_bubble);

	loop = g_main_loop_new (NULL, FALSE);
	g_timeout_add (1000, (GSourceFunc) stop_sd_main_loop, loop);
	g_main_loop_run (loop);

	g_assert_cmpint (g_sd_bubble_close, ==, TRUE);

	sd_bubble_clean_up (sd_bubble);
} */

static void
test_sd_bubble_move ()
{
	SdBubble* sd_bubble = sd_bubble_setup ();

	sd_bubble_move (sd_bubble, 10, 20);
	g_assert_cmpint (sd_bubble_get_x (sd_bubble), ==, 10);
	g_assert_cmpint (sd_bubble_get_y (sd_bubble), ==, 20);

	sd_bubble_clean_up (sd_bubble);
}

/* FIXME: these are based on interaction with a SD-bubble, so they don't really
** fit the concept of unit-tests */
/*static void
test_sd_bubble_is_visible ()
{
}

static void
test_sd_bubble_is_focused ()
{
}

static void
test_sd_bubble_is_hovered ()
{
}*/

/* FIXME: nothing but pain trying to implement a unit-test for the timeout */
/*static void
test_sd_bubble_start_timer ()
{
}

static void
timed_out_cb ()
{
	g_print ("%s() called\n", G_STRFUNC);
	g_sd_bubble_timed_out = TRUE;
}

static void
test_sd_bubble_timed_out ()
{
	SdBubble*  sd_bubble = sd_bubble_setup ();
	GMainLoop* loop      = NULL;

	g_signal_connect (sd_bubble, "timed-out", timed_out_cb, NULL);

	sd_bubble_start_timer (sd_bubble);
	loop = g_main_loop_new (NULL, FALSE);
	g_timeout_add (65 * 1000, (GSourceFunc) stop_sd_main_loop, loop);
	g_main_loop_run (loop);

	g_assert_cmpint (g_sd_bubble_timed_out, ==, TRUE);

	sd_bubble_clean_up (sd_bubble);
}*/

static void
test_sd_bubble_getset_dbus_sender ()
{
	SdBubble* sd_bubble = sd_bubble_setup ();
	gchar*    sender1   = ":1.1234";
	gchar*    sender2   = ":1.4321";

	sd_bubble_set_dbus_sender (sd_bubble, sender1);
	g_assert_cmpstr (sd_bubble_get_dbus_sender (sd_bubble), ==, sender1);
	sd_bubble_set_dbus_sender (sd_bubble, sender2);
	g_assert_cmpstr (sd_bubble_get_dbus_sender (sd_bubble), ==, sender2);

	sd_bubble_clean_up (sd_bubble);
}

static void
test_sd_bubble_getset_button_tint ()
{
	SdBubble* sd_bubble  = NULL;
	gchar*    actions[4] = {"ok_id", "Ok", "cancel_id", "Cancel"};

	g_settings = settings_new ();
	sd_bubble = sd_bubble_new (actions, 4, g_settings);
	g_assert (sd_bubble != NULL);
	g_assert (IS_SD_BUBBLE (sd_bubble));

	g_assert_cmpint (sd_bubble_get_button_tint (sd_bubble), ==, FALSE);
	sd_bubble_set_button_tint (sd_bubble, TRUE);
	g_assert_cmpint (sd_bubble_get_button_tint (sd_bubble), ==, TRUE);
	sd_bubble_set_button_tint (sd_bubble, FALSE);
	g_assert_cmpint (sd_bubble_get_button_tint (sd_bubble), ==, FALSE);

	g_object_unref (sd_bubble);
	settings_del (g_settings);
	g_settings = NULL;
}

GTestSuite *
test_sd_bubble_create_test_suite (void)
{
	GTestSuite *ts = NULL;
	
	ts = g_test_create_suite ("sd-bubble");

#define TC(x) g_test_create_case(#x, 0, NULL, NULL, x, NULL)

	g_test_suite_add (ts, TC (test_sd_bubble_new));
	g_test_suite_add (ts, TC (test_sd_bubble_getset_summary));
	g_test_suite_add (ts, TC (test_sd_bubble_getset_body));
	g_test_suite_add (ts, TC (test_sd_bubble_set_icon));
	g_test_suite_add (ts, TC (test_sd_bubble_get_num_actions));
	g_test_suite_add (ts, TC (test_sd_bubble_getset_actions));
	g_test_suite_add (ts, TC (test_sd_bubble_get_x));
	g_test_suite_add (ts, TC (test_sd_bubble_get_y));
	g_test_suite_add (ts, TC (test_sd_bubble_get_width));
	g_test_suite_add (ts, TC (test_sd_bubble_get_height));
	g_test_suite_add (ts, TC (test_sd_bubble_getset_state));
	g_test_suite_add (ts, TC (test_sd_bubble_getset_notification_id));
	g_test_suite_add (ts, TC (test_sd_bubble_show));
	/*g_test_suite_add (ts, TC (test_sd_bubble_close));*/
	g_test_suite_add (ts, TC (test_sd_bubble_move));
	/*g_test_suite_add (ts, TC (test_sd_bubble_is_visible));*/
	/*g_test_suite_add (ts, TC (test_sd_bubble_is_focused));*/
	/*g_test_suite_add (ts, TC (test_sd_bubble_is_hovered));*/
	/*g_test_suite_add (ts, TC (test_sd_bubble_start_timer));*/
	/*g_test_suite_add (ts, TC (test_sd_bubble_timed_out));*/
	g_test_suite_add (ts, TC (test_sd_bubble_getset_dbus_sender));
	g_test_suite_add (ts, TC (test_sd_bubble_getset_button_tint));

	return ts;
}
