#!/usr/bin/python3
# -*- coding: utf-8 -*-
#
# «oem-eula» - Presents a HW EULA to the user
#
# 
#
# Author:
#  - Kent Baxley <kent.baxley@canonical.com>

from ubiquity import misc
from ubiquity.plugin import *
import os
import glob
import gi
try:
    import ubunturecovery.recovery_common as magic
except ImportError:
    pass
    #this might fail if we didn't have ubuntu-recovery installed in some scenario
NAME = 'oem-eula'
AFTER = 'language'
BEFORE = 'timezone'
WEIGHT = 15

#Map of Ubuntu language to EULA representation of that language
MAP = {
    'ar'    : '817678-173',
    'bg'    : '817678-263', #* Bulgarian bg    
    'bs'    : '817678-FJ3',
    'cs'    : '817678-223', #* Czech cs
    'ca'    : '817678-003', ##* Catalan ca, Catala, Catalan language, replaced to English
    'da'    : '817678-083', #* Danish da
    'de'    : '817678-043', #* German de
    'el'    : '817678-153', #* Greek el
    'en'    : '817678-003', #* English en
    'es'    : '817678-E53', #* Spanish es
    'et'    : '817678-E43',
    'eu'    : '817678-003', ##* Basque eu, Euskara, Basque language, replaced to English 
    'fi'    : '817678-353', #* Finnish fi
    'fr_ca' : '817678-053', # replaced to fr_ww
    'fr'    : '817678-053', #* French fr
    'gl'    : '817678-003', ##* Galician gl, Galego, Galician language, replaced to English
    'he'    : '817678-BB3',
    'hr'    : '817678-BC3',
    'hu'    : '817678-213', #* Hungarian hu
    'id'    : '817678-BW3',
    'it'    : '817678-063', #* Italian it
    'ja'    : '817678-293', #* Japanese ja
    'kk'    : '817678-DF3',
    'ko'    : '817678-AD3', #* Korean ko
    'lt'    : '817678-E23',
    'lv'    : '817678-E13',
    'nl'    : '817678-333', #* Dutch nl
    'no'    : '817678-093',
    'pl'    : '817678-243', #* Polish pl
    'pt'    : '817678-133', #* Portuguese pt [rtf/pt_pt] 
    'pt_br' : '817678-203', ##* Portuguese (Brazil) pt_BR, Portugues do Brazil, Brazilian_Portuguese [rtf/pt_ww]
    'ro'    : '817678-273',
    'ru'    : '817678-253', #* Russian ru
    'sk'    : '817678-233', #* Slovak sk
    'sl'    : '817678-BA3',
    'sr'    : '817678-E33',
    'sv'    : '817678-103', #* Swedish sv
    'th'    : '817678-283',
    'tr'    : '817678-143',
    'uk'    : '817678-BD3',
    'vi'    : '817678-EP3',
    'zh_cn' : '817678-AA3', #* Chinese (Simplified) zh_CN
    'zh_tw' : '817678-AB3', #* Chinese (Traditional) zh_TW
}

EULA_DIRECTORY = '/usr/share/oem/eula/'

NM_BUS_NAME = 'org.freedesktop.NetworkManager'

class PageGtk(PluginUI):
    #plugin_title = 'ubiquity/text/eula_heading_label'

    def __init__(self, controller, *args, **kwargs):
        self.plugin_widgets = None
        self.controller = controller
        if 'UBIQUITY_OEM_USER_CONFIG' in os.environ:
            # WebKit2 spawns a process which we don't want to run as root
            misc.drop_privileges_save()

            gi.require_version('Gtk', '3.0')
            from gi.repository import Gtk
            builder = Gtk.Builder()
            builder.add_from_file('/usr/share/ubiquity/gtk/stepOEMEULA.ui')
            builder.connect_signals(self)
            self.controller.add_builder(builder)
            self.genuine_box = builder.get_object('genuine_box')
            self.accept_button = builder.get_object('accept_button')
            self.debug = builder.get_object('debug')
            self.plugin_widgets = builder.get_object('stepOEMEULA')
            eula_frame = builder.get_object('eula_frame')

            gi.require_version('WebKit2', '4.0')
            from gi.repository import WebKit2
            # We have no significant browsing interface, so there isn't much point
            # in WebKit creating a memory-hungry cache.
            context = WebKit2.WebContext.get_default()
            context.set_cache_model(WebKit2.CacheModel.DOCUMENT_VIEWER)

            self.webview = WebKit2.WebView()
            s = self.webview.get_settings()
            s.set_property('allow-file-access-from-file-urls', True)
            self.webview.connect('context-menu', self.on_context_menu)

            eula_frame.add(self.webview)
            eula_frame.show_all()

            #load the initial EULA page
            self.plugin_translate('C')

            if 'UBIQUITY_DEBUG' in os.environ:
                self.debug.show()

            # restore the previous permission
            misc.regain_privileges_save()

    def on_context_menu(self, unused_web_view, unused_context_menu,
                        unused_event, unused_hit_test_result):
        # True will not show the menu
        return True

    def plugin_translate(self, lang):
        #try to map it do a language if possible
        lang = lang.lower()
        if lang in MAP:
            lang = MAP[lang]
        text = 'file://'
        #check if we support this langauge
        file = glob.glob(EULA_DIRECTORY + '%s.html' % lang)
        if file:
            text += file[0]
        #We're guaranteed to have english
        else:
            text += EULA_DIRECTORY + MAP['en'] + '.html'
        self.webview.load_uri(text)

        #for helping to debug when a language isn't showing up
        self.debug.set_text(lang)

    def plugin_get_current_page(self):
        #If we have ubuntu-recovery installed, we can check the vendor
        #try:
        #    import ubunturecovery.recovery_common as magic
        #    if not magic.check_vendor():
        #        self.genuine_box.show()
        #        self.accept_button.set_sensitive(False)
        #except ImportError:
        #    pass

        if not self.accept_button.get_active():
            self.controller.allow_go_forward(False)
        return self.plugin_widgets

    def accept_toggled(self, widget):
        self.controller.allow_go_forward(widget.get_active())
