#include <linux/mmc/sdio_ids.h>
#include <linux/mmc/card.h>
#include <linux/mmc/host.h>
#include <linux/mmc/sdio.h>
#include <linux/mmc/core.h>
#include "sdio_io_ext.h"

/**
 * sdio_memcpy_toio_ext() - send sg list to sdio device
 * @func: sdio function
 * @addr: target addr to write to
 * @blocks: the total number of data blocks to copy
 * @blksz: the size of a block
 * @sg: the sg list of the source buffers
 * @sg_len: the length of the sg list
 * @op_code_fixed: specifies if the write should be done to a fixed
 * or incrementing address (according to SDIO cmd53 specification)
 *
 * Return value indicates if the transfer succeeded or not.
 *
 * This is an extension of the standard sdio_memcpy_toio function,
 * which can accept a sg list of any size instead of a single buffer.
 * This code is based on sdio_io_rw_ext_helper from mmc core.
 *
 * FIXME: This function contains code from MMC stack, need to verify
 * there's no any licensing issue here.
 */
int sdio_memcpy_toio_ext(struct sdio_func *func, unsigned int addr,
			 unsigned blocks, unsigned blksz,
			 struct scatterlist sg, size_t sg_len,
			 bool op_code_fixed)
{
	struct mmc_request mrq = {0};
	struct mmc_command cmd = {0};
	struct mmc_data data = {0};

	if (WARN_ON(!func->card))
		return -EINVAL;
	if (WARN_ON(func->num > 7))
		return -EINVAL;
	if (WARN_ON(blocks == 1 && blksz > 512))
		return -EINVAL;
	if (WARN_ON(blocks == 0))
		return -EINVAL;
	if (WARN_ON(blksz == 0))
		return -EINVAL;

	/* sanity check, according to the spec, valid range is [0x0:0x1ffff] */
	if (addr & ~0x1FFFF)
		return -EINVAL;

	mrq.cmd = &cmd;
	mrq.data = &data;

	cmd.opcode = SD_IO_RW_EXTENDED;
	cmd.arg = 0x80000000;
	cmd.arg |= func->num << 28;
	cmd.arg |= op_code_fixed ? 0x04000000 : 0x00000000;
	cmd.arg |= addr << 9;
	if (blocks == 1 && blksz <= 512)
		cmd.arg |= (blksz == 512) ? 0 : blksz;	/* byte mode */
	else
		cmd.arg |= 0x08000000 | blocks;		/* block mode */
	cmd.flags = MMC_RSP_SPI_R5 | MMC_RSP_R5 | MMC_CMD_ADTC;

	data.blksz = blksz;
	data.blocks = blocks;
	data.flags = MMC_DATA_WRITE;
	data.sg = &sg;
	data.sg_len = sg_len;

	mmc_set_data_timeout(&data, card);
	mmc_wait_for_req(card->host, &mrq);

	if (cmd.error)
		return cmd.error;
	if (data.error)
		return data.error;

	if (cmd.resp[0] & R5_ERROR)
		return -EIO;
	if (cmd.resp[0] & R5_FUNCTION_NUMBER)
		return -EINVAL;
	if (cmd.resp[0] & R5_OUT_OF_RANGE)
		return -ERANGE;

	return 0;
}
