/******************************************************************************
 *
 * Copyright(c) 2005 - 2013 Intel Corporation.
 * All rights reserved.
 *
 * LICENSE PLACE HOLDER
 *
 *****************************************************************************/

#ifndef __IWL_EMULATION_H__
#define __IWL_EMULATION_H__

#include <linux/types.h>
#include <linux/sched.h>
#include <linux/interrupt.h>
#include "trans_slave/idi_al.h"

struct iwl_al_t;
struct fw_img;
struct iwl_trans;

/*
 * Logical channels define flags:
 */
#define IWL_IDI_CHANNEL_START		0

/* temp : To assure Trans that inta addition is used */
#define IWL_IDI_INTA

/*numbers of  DBB Channels */
#define IWL_IDI_MAX_CHANNELS	3

/* SG list command flags for DMA */
#define IWL_DMA_CTRL_MOVE_NO_INT			0x0
#define IWL_DMA_CTRL_MOVE_INT				0x1
#define IWL_DMA_CTRL_MOVE_LAST_NO_INT		0x2
#define IWL_DMA_CTRL_MOVE_LAST_AND_INT		0x3

/* Debugfs directory for the emulation */
#define IWL_EMULATION_DEBUGFS_DIR			"iwl_emulation_dbg"

/* Define which module of the IDI runs on which processor */
#define IWL_IDI_TX_WORK_CPU 0
#define IWL_IDI_RX_WORK_CPU 1

/*
 * GLobal AL Emulation struct.
 */
extern struct iwl_al_t *iwl_al_em;

/*
 * Global variables that marks the start address of the AL sram
 */
extern u32 AL_SRAM_ADDRESS;
extern void *AL_SRAM_VIRTUAL_ADDRESS;

/**
 *  Describes the amfh configuration that should be used by the al
 *
 *  @max_burst_size: maximal number of frames in a burst
 *  @rb_size: rb size in the RX ring
 *  @num_rbds: size of the RX ring
 */
struct iwl_al_amfh_config {
	u32 max_burst_size;
	u32 rb_size;
	u32 num_rbds;
};

/**
 *  Describes the Tx configuration
 *
 *  @num_tx_queues: number of tx queues (including the command queues)
 *  @num_tx_tfds: number of slots in the transmit queues.
 *  @num_cmd_tfds: number of slots in the command queues.
 *  @num_tfds: number of overall tfds in the system
 *  @tb_size: size of transmit buffer, 256 or 512 bytes.
 *  @num_internal_tfds: number of tfds for internal use.
 *  @num_internal_tb: number of transmit buffers for internal use.
 */
struct iwl_al_tx_config {
	u32 num_tx_queues;
	u32 num_tx_tfds;
	u32 num_cmd_tfds;
	u32 num_tfds;
	u32 tb_size;
	u32 num_internal_tfds;
	u32 num_internal_tb;
};

/*
 * Callback for the interrupt handler. The callback should be done in an
 * aync. manner.
 * In the context of the emulation, need to consider the correct context for
 * calling this function.
 */
typedef int (*iwl_al_irq) (unsigned long);

/**
  * Configuration of the al block
  *
  *  @amfh_config: amfh configuration
  *  @tx_config: tx configuration.
  */
struct iwl_al_config {
	struct iwl_al_amfh_config amfh_config;
	struct iwl_al_tx_config tx_config;
	irqreturn_t (*iwl_al_irq_handler) (unsigned long);
	irqreturn_t (*iwl_al_irq_thread) (unsigned long);
};

/* Error flags */
#define EINVCHAN		35 /* Channel out of range */
#define EDMAENABLED		36 /* Illegal action while DMA is enabled */
#define EDMADISABLED		37 /* Illegal action while DMA is enabled */
#define ENOHNDLR		38 /* No habdler for IRQ call */
#define EHWMISUSE		39 /* Using the HW in an erronneous way */

/*****************************************************************************
 *
 * System flows
 *
 *****************************************************************************/

/**
 * Initialize the emulation block.
 *
 * @config: emulation module configuration
 * @trans: the transport layer
 * Returns 0 on success, negative value describing the error otherwise.
 */
int iwl_al_init(struct iwl_al_config *config, struct iwl_trans *trans);

/**
 * Free the emulation block.
 * Before calling free the block should be stopped.
 */
void iwl_al_free(void);

/**
 * start  the emulation block. The start should be called only immedialtly
 * after init flow, or after stop flow. Starting the block while it is already
 * started will fail with an error return value.
 *
 * Returns 0 on success, negative value describing the error otherwise.
 */
int iwl_al_start(void);

/**
 * Stop the emulation block.
 * Once the stop flow is complete the block as should be back in its initial
 * state (as after init and before start was called).
 * Stopping a block when it is already stopped will return immediatly with
 * return value indicating success.
 *
 * Returns 0 on success, negative value describing the error otherwise.
 */
int iwl_al_stop(void);

/**
 * Load a given LMAC FW to the device
 *
 * @image: the image to load
 */
int iwl_al_load_given_ucode(struct fw_img *image);

/******************************************************************************
 *
 * IDI interface operations. These include: SG DMA and Interrupt forwarding.
 * Remote register access will be covered buy the IDI bus emulation.
 *
 *****************************************************************************/

#define IWL_IDI_DBB_TX 0x0
#define IWL_IDI_DBB_RX 0x1

/*
 * Callback for the IDI DBB DMA interrupt. The callback must always be called in
 * an async. manner.
 * In the context of the emulation, need to consider the correct context for
 * calling this function.
 *
 * @chan: the channel
 * @dir: the direction, IWL_IDI_DBB_TX or IWL_IDI_DBB_RX
 * @irq_handler: the function pointer to the irq_handler
 * @data: A void pointer to data that will be sent to the interrupt handler
 *	when it is called.
*/
typedef void (*iwl_idi_al_dbb_dma_irq) (__le32 chan, __le32 dir, void *data);

/**
 *  Set the interrupt handler for the given channel, in the given direction
 *  The call should be done by the driver transport layer.
 *
 * @chan: the channel
 * @dir: the direction, IWL_IDI_DBB_TX or IWL_IDI_DBB_RX
 * @irq_handler: the function pointer to the irq_handler
 * @data: A void pointer to data that will be sent to the interrupt handler
 *	 when it is called.
 *
 *  Returns 0 on success, negative value describing the error otherwise.
 */
int iwl_al_dbb_dma_set_irq_handler(__le32 chan, __le32 dir,
				   iwl_idi_al_dbb_dma_irq irq_handler,
				   void *data);

/**
 * Descriptor of the DBB dma block
 *
 * @next: [31..4] Next linked node address. A value of zero is interrupted as
 *            NIL (this will stop the list processing
 *            (this means that the address must be 16 bytes aligned)
 *            [3..2] not used
 *            [1] Node interrupt enable
 *            [0] Last node information
 *
 * @base: [31..2] Base address (means that the buffer must be 4 bytes aligned)
 * @size: [31..20] Reserved
 *           [19..2] The size in 32 bits words. The value 0 is not allowed.
 *           [1..0] reserved
 */
struct idi_sg_desc {
	__le32 next;
	__le32 base;
	__le32 size;
	__le32 pad;
};

/**
 *  Set a scatter/gather list in the given channel, in the given direction.
 *  The call should be done by the driver transport layer.
 *  The call should be done only when the <chan, dir> is stopped.
 *
 *  @al:reference to struct iwl_al_t
 *  @chan: the channel
 *  @dir: the direction, IWL_IDI_DBB_TX or IWL_IDI_DBB_RX
 *  @sg_ll: scatter/gather list
 *
 *  Returns 0 on success, negative value describing the error otherwise.
 */
int iwl_al_dbb_set_sg(__le32 chan, __le32 dir,
		      struct idi_sg_desc *sg_ll);

/**
 *  Start the DBB DMA operation on the given channel, in the given direction.
 *  The DMA operation should be handled asyncronuously.
 *  The call should be done by the driver transport layer after it configured
 *  the sg list
 *
 *  @al:reference to struct iwl_al_t
 *  @chan: the channel
 *  @dir: the direction, IWL_IDI_DBB_TX or IWL_IDI_DBB_RX
 *
 *  Returns 0 on success, negative value describing the error otherwise.
 */
int iwl_al_dbb_start(__le32 chan, __le32 dir);

/**
 *  Stop the DBB DMA the given channel, in the given direction.
 *  The call is syncronuous, and on return the DMA is halted.
 *  The call should be done by the driver transport layer.
 *
 *  @chan: the channel
 *  @dir: the direction, IWL_IDI_DBB_TX or IWL_IDI_DBB_RX
 *
 *  Returns 0 on success, negative value describing the error otherwise.
 */
int iwl_al_dbb_stop(__le32 chan, __le32 dir);

#define IWL_IDI_ABB_TX_HP_ADDR	0x00020000
#define IWL_IDI_ABB_TX_LP_ADDR	0x00028000
#define IWL_IDI_ABB_RX_ADDR		0x00030000

/**
 *  Set the address of the ABB buffer for the given channel, in the given
 *  direction.
 *  The call should be done by the driver transport layer.
 *
 *  @chan: the channel
 *  @dir: the direction, IWL_IDI_DBB_TX or IWL_IDI_DBB_RX
 *  @addr: dummy address as specified above.
 *
 *  Returns 0 on success, negative value describing the error otherwise.
 */
int iwl_al_abb_config(__le32 chan, __le32 dir, __le32 addr);

/**
 *  Enable the AL interrupt.
 *
 * Returns 0 on success, negative value describing the error otherwise.
 */
int iwl_al_irq_enable(void);

/**
 *  Disable the AL interrupt.
 *
 * Returns 0 on success, negative value describing the error otherwise.
 */
int iwl_al_irq_disable(void);

/* Prints debug data on error */
void iwl_al_print_error_data(void);

/*
 * Returns a pointer to the emulation debugfs directory.
 */
struct dentry *iwl_emulation_get_debugfs_dir(void);

/******************************************************************************
 *
 * Emulation Logging
 *
 *****************************************************************************/

/**
 *  Logging functions - macros for logging in emulation.
 *  These macros should be encapsulated in each module, which will provide
 *  a suitable prefix (e.g. [AMFH]) and level. The levels should come
 *  from DEV_CRIT, DEV_INFO and the others, where logging level can
 *  easily changed statically. See iwl-em-amfh.c for an exmaple.
 **/
#ifdef IWL_EMULATION_LOG_ENABLED
#define IWL_EM_TRACE_ENTER(level, prefix) \
	printk(level "Emulation: %s >>> %s, thread %d\n",  prefix, \
					__func__, get_current()->tgid);
#define IWL_EM_TRACE_EXIT(level, prefix) \
	printk(level "Emulation: %s <<< %s , thread %d\n",  prefix, \
					__func__, get_current()->tgid);
#define IWL_EM_TRACE_EXIT_RET(level, prefix, ret) \
	printk(level "Emulation: %s <<< %s (%d)\n", prefix, __func__, ret)
#define IWL_EM_TRACE_EXIT_PTR(level, prefix, ptr) \
	printk(level "Emulation: %s <<< %s (%p)\n", prefix, __func__, ptr)
#define IWL_EM_LOG(level, prefix, fmt, args ...) \
	printk(level "Emulation: %s %s - " fmt "\n", prefix, __func__, ## args)
#define IWL_EM_LOG_HEX_DUMP(msg, p, len) \
	do { \
		printk(KERN_DEBUG msg); \
		print_hex_dump(KERN_DEBUG, "", \
			DUMP_PREFIX_ADDRESS, 32, 4, p, len, 0); \
	} while (0)

#else

#define IWL_EM_TRACE_ENTER(level, prefix)
#define IWL_EM_TRACE_EXIT(level, prefix)
#define IWL_EM_TRACE_EXIT_RET(level, prefix, ret)
#define IWL_EM_TRACE_EXIT_PTR(level, prefix, ptr)
#define IWL_EM_LOG(level, prefix, fmt, args ...)
#define IWL_EM_LOG_HEX_DUMP(msg, p, len)

#endif
/* Always print error messages */
#define IWL_EM_LOG_ERR(fmt, arg ...) \
	WARN(1, "Emulation: %s : " fmt,  __func__,  ## arg)

#endif /* __IWL_EMULATION_H__ */
