/******************************************************************************
 *
 * Copyright(c) 2005 - 2013 Intel Corporation.
 * All rights reserved.
 *
 * LICENSE PLACE HOLDER
 *
 *****************************************************************************/

#ifndef _IWL_SFDB_H__
#define _IWL_SFDB_H__

#define SFDB_NUM_QUEUES	(16)
#define SFDB_TFD_NUM_ENTRIES	(32)
#define SFDB_SCD_QUEUE_SIZE	(256)

#define SFDB_INVALID_ENTRY	(0xFF)

#define SFDB_NUM_OF_TBS	(20)
#define SFDB_TB_SIZE		(256)

/* Byte count table base address allocated in the sfdb and
 * exported with global*/
extern u32 IWL_SFDB_BC_ADDRESS;

/**
 * Initializes the SFDB module in the emulation.
 *
 *@trans - transport to the pci.
 *
 * Returns 0 on success, negative value describing the error otherwise.
 */
int iwl_sfdb_em_init(struct iwl_trans *trans);

/*
 * Start  the SFDB.
 * The start should be called only after sfdb init or after stop flow.
 *
 * Returns 0 on success, negative value describing the error otherwise.
 */
int iwl_sfdb_em_start(void);

/*
 * Stops the SFDB.
 * Does not free the data structures just zeroes them.
 *
 * NOTE: The LMAC should be stopped before this call to prevent
 * a race condition of LMAC pulling data from zeroed ring.
 *
 * Returns 0 on success, negative value describing the error otherwise.
 */
int iwl_sfdb_em_stop(void);

/**
 * Frees the sfdb emulation struct.
 *
 * The al emulation stopping should be called before trying to stop the SFDB.
 */
void iwl_sfdb_em_free(void);

/**
 * This function initializes a TFD queue and configures FH with the base
 * address of this queue.
 * Allocates a DMA memory for the PCIe TFD queue.
 *
 * @param queue_id - the number of queue.
 */
void iwl_sfdb_em_init_queue(u8 queue_id);

/**
 * Writes the given value to the queues tfd entry in the Byte Count table.
 *
 * @ param queue_id - The queue id to update the byte count value.
 * @ param tfd_entry - The tfd entry in the queue to which we update the bc.
 * @ param value - The value to write to BC_TABLE[queue_id][tfd_entry]
 */
void iwl_sfdb_em_write_bc(u8 queue_id, u8 tfd_entry, __le16 value);

/**
 * Writes the given value to the LUT table in the tfd entry.
 *
 * @ param queue_id - The queue id to update the LUT value.
 * @ param tfd_entry - The tfd entry in the queue to which we update the LUT.
 * @ param value - The value to write to LUT_TABLE[queue_id][tfd_entry]
 */
void iwl_sfdb_em_write_lut_value(u8 queue_id, u8 tfd_entry, u16 value);

/**
 * Frees all TFDs in the queue up to (but not including) given tfd_entry.
 * Note that tfd_entry is a pointer to the queue (like the read pointer),
 * not an index in the TFD pool.
 *
 * @param queue_id - The number of queue.
 * @param tfd_index - The index of the TFD in the TFD pool
 */
void iwl_sfdb_em_release_tfds(u8 queue_id, u8 tfd_entry);

/**
 * Sets the next free entry in the lut of the queue
 * to point to IDI TFD index from the pool.
 *
 * @param queue_id - The number of queue.
 * @param tfd_index - The index of the TFD in the TFD pool
 *
 * @return void
 */
void iwl_sfdb_em_attach_tfd(u8 queue_id, u8 tfd_index);

/**
 * This function deattach TFD from the queue.
 * Because we use a PCIe queue, there is no "de-attaching" as removing the
 * TFD, but only invalidating the relevant index in tfd_indices (telling that no
 * TFD in the pool now referenced to that slot in the queue).
 *
 * @param U08 : queueId  -- the number of queue
 * @param U08 : tfdEntry -- the entry in the TFD queue
 */
void iwl_sfdb_em_deattach_tfd(u8 queue_id, u8 tfd_entry);

/**
 * Increments the write pointer and updates the SCD register.
 *
 * @param queueId - The number of queue.
 */
void iwl_sfdb_em_inc_write_ptr(u8 queue_id);

/**
 * Gets the write pointer of the given queue.
 *
 * @param queueId - The number of queue.
 *
 * Returns the write pointer of the queue.
 */
u32  iwl_sfdb_em_get_write_ptr(u8 queue_id);

/**
 * Update Write pointer of Scheduler for queueId
 *
 * @param queueId - The number of queue
 * @param index - The index to set the write pointer
 */
void iwl_sfdb_em_hw_set_write_ptr(u8 queue_id, u32 index);

/**
 * Returns the read pointer of the queue
 *
 * @param queue_id - the number of queue.
 *
 * Returns the read pointer of the given queue.
 */
u8 iwl_sfdb_em_get_read_ptr(u8 queue_id);

/**
 * Update read pointer of Scheduler for queueId
 *
 * @param queue_id - The number of queue
 * @param index - The index to set the read pointer
 */
void iwl_sfdb_em_hw_set_read_ptr(u8 queue_id, u8 index);

/**
 * This function updates the byte count table for this queue and index
 *
 * @param queueId - the number of queue
 * @param byte_cnt - the number of bytes for this frame
 *
 * @return void
 */
void iwl_sfdb_em_update_byte_count(u8 queue_id, __le16 byte_cnt);

/**
 * Configures the LMAC (FH/SCD) for using this queue
 *
 * @param queueId - The number of queue
 */
void iwl_sfdb_em_config_hw(u8 queue_id);

/**
 * This function sets the context data for the queue
 *
 * @queue_id - the number of queue
 *
 * @return void
 */
void iwl_sfdb_em_set_ctx_data(u8 queue_id, u32 scd_base_addr);

/**
 * This function sets the status of this queue (to SCD status register)
 *
 * @queueId - the number of queue
 * @isActive - is the queue active
 * @isPan - is the queue for PAN
 * @txFifo - the Q->Fifo mapping
 *
 * @return void
 */
void iwl_sfdb_em_set_status(u8 queue_id, u32 is_active,
			    u32 is_pan, u32 tx_fifo);

/**
 * This function returns the status of this queue (from SCD status register)
 *
 * @queueId- The number of queue.
 */
u32 iwl_sfdb_em_get_status(u8 queue_id);

/**
 * Returns if the queue is aggregated.
 *
 * @param queue_id - The number of queue
 *
 * Returns TRUE if the queue is active, FALSE otherwise.
 */
bool iwl_sfdb_em_queue_is_aggregated(u8 queue_id);

#endif /* _IWL_SFDB_H__ */
