/* Copyright 2011 Canonical, Ltd. This software is licensed under the GNU
 * Lesser General Public License version 3 or later (see the file COPYING).
 */

#ifndef OIF_QML_GESTUREEVENT_H_
#define OIF_QML_GESTUREEVENT_H_

#include <QDeclarativeItem>

#include "gesture.h"

class Touch;
class Device;

/**
 * An item that provides information about a gesture event.
 */

class GestureEvent : public QDeclarativeItem {
  Q_OBJECT
  /**
   * Setting accepted to false in the onGestureBegin handler causes the
   * gesture to be rejected.
   *
   * Other GestureArea items may accept the event, or the event may be
   * rejected entirely, and the touch events may be reprocessed by the system.
   * This property defaults to true.
   */
  Q_PROPERTY(bool accepted READ accepted WRITE set_accepted NOTIFY
             acceptedChanged)

  /**
   * This property holds the list of touches comprising the gesture.
   */
  Q_PROPERTY(const QDeclarativeListProperty< Touch > touches READ touches)

  /**
   * This property holds a reference to the gesture subscription.
   */
  Q_PROPERTY(const Gesture* gesture READ gesture CONSTANT)

  /**
   * This property holds the ID of the gesture event.
   */
  Q_PROPERTY(int id READ id CONSTANT)

  /**
   * This property holds the information of the device.
   */
  Q_PROPERTY(Device* device READ device CONSTANT)

  /**
   * This property holds a variant map of attributes of the gesture.
   *
   * All gesture attributes are available in this property. The keys may be
   * referenced in the geis.h header file.
   */
  Q_PROPERTY(const QVariantMap attributes READ attributes CONSTANT)

 public:
  GestureEvent() {} /* To satisfy QML */
  GestureEvent(GestureArea* area,
               GeisFrame frame,
               GeisTouchSet touch_set,
               Device* device);

  bool accepted() const { return accepted_; }
  void set_accepted(bool new_value);
  const Gesture* gesture() { return gesture_; }
  int id() const { return id_; }
  Device* device() { return device_; }
  const QVariantMap attributes() const { return attributes_; }
  QDeclarativeListProperty< Touch > touches() {
    return QDeclarativeListProperty< Touch >(this, touches_);
  }

 signals:
  void acceptedChanged();

 private:
  int id_;
  Gesture* gesture_;
  QList< Touch* > touches_;
  QVariantMap attributes_;
  bool accepted_;
  Device* device_;

  Q_DISABLE_COPY(GestureEvent)
};

QML_DECLARE_TYPE(GestureEvent)

#endif  // OIF_QML_GESTUREEVENT_H_
