
/*
 * Copyright (C) 2004-2005 Maximilian Schwerin
 *
 * This file is part of oxine a free media player.
 * Most of this file was copied from the xine-ui project.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: download.c 1410 2006-10-21 14:19:43Z mschwerin $
 *
 */
#include "config.h"

#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>

#ifdef HAVE_CURL
#include <curl/curl.h>
#include <curl/types.h>
#include <curl/easy.h>
#endif

#include "heap.h"
#include "i18n.h"
#include "logger.h"
#include "mutex.h"
#include "download.h"
#include "environment.h"
#include "oxine.h"

#ifdef HAVE_CURL

extern oxine_t *oxine;

#define USER_AGENT  "User-Agent: "PACKAGE_NAME"/"PACKAGE_VERSION
#define USER_PWD    "anonymous:"PACKAGE_BUGREPORT

static int
progress_callback (void *userdata,
                   double dltotal, double dlnow, double ultotal, double ulnow)
{
    download_t *download = (download_t *) userdata;

#ifdef DEBUG
    int percent = (dltotal > 0.0) ? (int) (dlnow * 100.0 / dltotal) : 0;
    debug ("downloading %d %%", percent);
#endif

    /* return non 0 abort transfert */
    return download->status;
}

static int
store_data (void *ptr, size_t size, size_t nmemb, void *userdata)
{
    download_t *download = (download_t *) userdata;
    int rsize = size * nmemb;

    if (download->size == 0)
        download->buf = (char *) ho_malloc ((rsize + 1));
    else
        download->buf = (char *) ho_realloc (download->buf,
                                             download->size + rsize + 1);

    if (download->buf) {
        memcpy (&(download->buf[download->size]), ptr, rsize);
        download->size += rsize;
        download->buf[download->size] = 0;
    } else
        download->status = 1;

    return rsize;
}
#endif

int
network_download (const char *url, download_t * download)
{
#ifdef HAVE_CURL
    debug ("Starting download of %s", url);

    CURL *curl;
    CURLcode res;

    mutex_lock (&oxine->download_mutex);

    curl_global_init (CURL_GLOBAL_DEFAULT);

    if ((curl = curl_easy_init ()) != NULL) {
        char error_buffer[CURL_ERROR_SIZE + 1];
        memset (&error_buffer, 0, sizeof (error_buffer));

#ifdef DEBUG
        int verbose = 1;
#else
        int verbose = 0;
#endif

        curl_easy_setopt (curl, CURLOPT_VERBOSE, verbose);
        curl_easy_setopt (curl, CURLOPT_URL, url);
        curl_easy_setopt (curl, CURLOPT_USERAGENT, USER_AGENT);
        curl_easy_setopt (curl, CURLOPT_NOSIGNAL, 1);
        curl_easy_setopt (curl, CURLOPT_FAILONERROR, 1);
        curl_easy_setopt (curl, CURLOPT_NETRC, 1);
        curl_easy_setopt (curl, CURLOPT_FOLLOWLOCATION, 1);

        curl_easy_setopt (curl, CURLOPT_WRITEFUNCTION, store_data);
        curl_easy_setopt (curl, CURLOPT_WRITEDATA, (void *) download);

        curl_easy_setopt (curl, CURLOPT_NOPROGRESS, 0);
        curl_easy_setopt (curl, CURLOPT_PROGRESSFUNCTION, progress_callback);
        curl_easy_setopt (curl, CURLOPT_PROGRESSDATA, (void *) download);

        curl_easy_setopt (curl, CURLOPT_ERRORBUFFER, &error_buffer);
        curl_easy_setopt (curl, CURLOPT_USERPWD, USER_PWD);

        if ((res = curl_easy_perform (curl)) != 0) {
            download->error = ho_strdup ((strlen (error_buffer))
                                         ? error_buffer : "Unknown error");
            download->status = 1;
        }

        curl_easy_cleanup (curl);
    }

    curl_global_cleanup ();

    mutex_unlock (&oxine->download_mutex);

    return (download->status == 0);
#else
    return 0;
#endif
}


download_t *
download_new (void)
{
    download_t *download = ho_new (download_t);

    download->buf = NULL;
    download->error = NULL;
    download->size = 0;
    download->status = 0;

    return download;
}


void
download_free (download_t * download)
{
    if (download->buf)
        ho_free (download->buf);
    if (download->error)
        ho_free (download->error);
    if (download)
        ho_free (download);
}


char *
download_to_cache (const char *url, const char *filename, int force_update)
{
#ifdef HAVE_CURL
    /* Create the URL of the cache. */
    int prefix_len = 0;
    if (strncasecmp (url, "http://", 7) == 0)
        prefix_len = 7;
    else if (strncasecmp (url, "ftp://", 6) == 0)
        prefix_len = 6;
    else {
        error (_("Unknown prefix to URL '%s'!"), url);
        return NULL;
    }

    char cache_url[1024];
    if (filename)
        snprintf (cache_url, 1024, "%s/%s", get_dir_oxine_tmp (), filename);
    else
        snprintf (cache_url, 1024, "%s/%s",
                  get_dir_oxine_tmp (), (url + prefix_len));

    /* If the file is already in the cache we quit. */
    if (!force_update && (access (cache_url, R_OK) == 0))
        return ho_strdup (cache_url);

    /* Create the cache directory */
    char *dirn = get_dirname (cache_url);
    if (!mkdir_recursive (dirn, 0700)) {
        ho_free (dirn);
        return NULL;
    }
    ho_free (dirn);

    /* Download the file. */
    download_t *download = download_new ();
    if (!network_download (url, download)) {
        download_free (download);
        return NULL;
    }

    /* Save the file to the cache. */
    int fd = open (cache_url,
                   O_CREAT | O_WRONLY | O_TRUNC);
    if (fd == -1) {
        error (_("Could not create '%s': %s!"), cache_url, strerror (errno));
        download_free (download);
        return NULL;
    }

    if (write (fd, download->buf, download->size) == -1) {
        error (_("Could not write to '%s': %s!"), cache_url,
               strerror (errno));
        download_free (download);
        close (fd);
        return NULL;
    }

    fchmod (fd, S_IRUSR | S_IWUSR);
    close (fd);
    download_free (download);

    return ho_strdup (cache_url);
#else
    return NULL;
#endif
}


int
is_downloadable (const char *url)
{
    if (url) {
        if ((strncasecmp (url, "http://", 7) == 0)
            || (strncasecmp (url, "ftp://", 6) == 0))
            return 1;
    }
    return 0;
}
