
/*
 * Copyright (C) 2004-2005 Maximilian Schwerin
 *
 * This file is part of oxine a free media player.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: odk_keymap.c 1367 2006-10-15 19:24:20Z mschwerin $
 *
 */

#include "config.h"

#include <string.h>
#include <X11/Xlib.h>
#include <X11/keysym.h>

#include "event.h"
#include "logger.h"
#include "odk_keymap.h"

typedef struct {
    KeySym kbd_key;             /* Keyboard key. */
    int kbd_modifier;           /* Keyboard modifier. */
    char *lirc_action;          /* LIRC action. */

    oxine_key_id_t key_id;      /* Internal key ID. */
    int how;

    char *description;
} odk_keymap_entry_t;

static odk_keymap_entry_t default_keymap_table[] = {
    /* Menus */
    {XK_F1, 0, "menu1", OXINE_KEY_MENU1, 0, NULL},
    {XK_F2, 0, "menu2", OXINE_KEY_MENU2, 0, NULL},
    {XK_F3, 0, "menu3", OXINE_KEY_MENU3, 0, NULL},
    {XK_F4, 0, "menu4", OXINE_KEY_MENU4, 0, NULL},
    {XK_F5, 0, "menu5", OXINE_KEY_MENU5, 0, NULL},
    {XK_F6, 0, "menu6", OXINE_KEY_MENU6, 0, NULL},
    {XK_F7, 0, "menu7", OXINE_KEY_MENU7, 0, NULL},

    /* Numbers */
    {XK_0, 0, "0", OXINE_KEY_0, 0, NULL},
    {XK_KP_0, 0, "0", OXINE_KEY_0, 0, NULL},
    {XK_1, 0, "1", OXINE_KEY_1, 0, NULL},
    {XK_KP_1, 0, "1", OXINE_KEY_1, 0, NULL},
    {XK_2, 0, "2", OXINE_KEY_2, 0, NULL},
    {XK_KP_2, 0, "2", OXINE_KEY_2, 0, NULL},
    {XK_3, 0, "3", OXINE_KEY_3, 0, NULL},
    {XK_KP_3, 0, "3", OXINE_KEY_3, 0, NULL},
    {XK_4, 0, "4", OXINE_KEY_4, 0, NULL},
    {XK_KP_4, 0, "4", OXINE_KEY_4, 0, NULL},
    {XK_5, 0, "5", OXINE_KEY_5, 0, NULL},
    {XK_KP_5, 0, "5", OXINE_KEY_5, 0, NULL},
    {XK_6, 0, "6", OXINE_KEY_6, 0, NULL},
    {XK_KP_6, 0, "6", OXINE_KEY_6, 0, NULL},
    {XK_7, 0, "7", OXINE_KEY_7, 0, NULL},
    {XK_KP_7, 0, "7", OXINE_KEY_7, 0, NULL},
    {XK_8, 0, "8", OXINE_KEY_8, 0, NULL},
    {XK_KP_8, 0, "8", OXINE_KEY_8, 0, NULL},
    {XK_9, 0, "9", OXINE_KEY_9, 0, NULL},
    {XK_KP_9, 0, "9", OXINE_KEY_9, 0, NULL},

    /* Zoom */
    {XK_z, ControlMask | Mod1Mask, NULL, OXINE_KEY_ZOOM_RESET, 0, NULL},
    {XK_y, ControlMask | Mod1Mask, NULL, OXINE_KEY_ZOOM_RESET, 0, NULL},
    {XK_Z, ControlMask | Mod1Mask, NULL, OXINE_KEY_ZOOM_RESET, 0, NULL},
    {XK_Y, ControlMask | Mod1Mask, NULL, OXINE_KEY_ZOOM_RESET, 0, NULL},
    {XK_z, ControlMask, NULL, OXINE_KEY_ZOOM_X, +1, NULL},
    {XK_y, ControlMask, NULL, OXINE_KEY_ZOOM_X, +1, NULL},
    {XK_Z, ControlMask, NULL, OXINE_KEY_ZOOM_X, -1, NULL},
    {XK_Y, ControlMask, NULL, OXINE_KEY_ZOOM_X, -1, NULL},
    {XK_z, Mod1Mask, NULL, OXINE_KEY_ZOOM_Y, +1, NULL},
    {XK_y, Mod1Mask, NULL, OXINE_KEY_ZOOM_Y, +1, NULL},
    {XK_Z, Mod1Mask, NULL, OXINE_KEY_ZOOM_Y, -1, NULL},
    {XK_Y, Mod1Mask, NULL, OXINE_KEY_ZOOM_Y, -1, NULL},
    {XK_z, 0, NULL, OXINE_KEY_ZOOM, +1, NULL},
    {XK_y, 0, NULL, OXINE_KEY_ZOOM, +1, NULL},
    {XK_Z, 0, NULL, OXINE_KEY_ZOOM, -1, NULL},
    {XK_Y, 0, NULL, OXINE_KEY_ZOOM, -1, NULL},

    /* Audio Channel, Audio Offset */
    {XK_x, 0, "audio_channel", OXINE_KEY_AUDIO_CHANNEL, +1, NULL},
    {XK_X, 0, "audio_channel", OXINE_KEY_AUDIO_CHANNEL, -1, NULL},
    {XK_c, 0, "audio_offset", OXINE_KEY_AUDIO_OFFSET, +9, NULL},
    {XK_C, 0, "audio_offset", OXINE_KEY_AUDIO_OFFSET, -9, NULL},

    /* Subtitle Channel, Subtitle Offset */
    {XK_r, 0, "spu_channel", OXINE_KEY_SPU_CHANNEL, +1, NULL},
    {XK_R, 0, "spu_channel", OXINE_KEY_SPU_CHANNEL, -1, NULL},
    {XK_t, 0, "spu_offset", OXINE_KEY_SPU_OFFSET, +9, NULL},
    {XK_T, 0, "spu_offset", OXINE_KEY_SPU_OFFSET, -9, NULL},

    /* Volume */
    {XK_v, 0, "volume", OXINE_KEY_VOLUME, +5, "increase volume by 5%"},
    {XK_plus, 0, "volume", OXINE_KEY_VOLUME, +5, "increase volume by 5%"},
    {XK_KP_Add, 0, "volume", OXINE_KEY_VOLUME, +5, "increase volume by 5%"},

    {XK_V, 0, "volume", OXINE_KEY_VOLUME, -5, "decrease volume by 5%"},
    {XK_minus, 0, "volume", OXINE_KEY_VOLUME, -5, "decrease volume by 5%"},
    {XK_KP_Subtract, 0, "volume", OXINE_KEY_VOLUME, -5, "decrease volume by 5%"},

    {XK_m, 0, "volmute", OXINE_KEY_VOLMUTE, 0, "mute volume"},

    /* Stream control */
    {0, 0, "play", OXINE_KEY_PLAY, 0, "play"},
    {0, 0, "pause", OXINE_KEY_PAUSE, 0, "pause"},
    {XK_Pause, 0, "pplay", OXINE_KEY_PPLAY, 0, "play/ pause"},

    {XK_s, 0, "stop", OXINE_KEY_STOP, 0, "stop"},
    {XK_S, 0, "stop", OXINE_KEY_STOP, 0, "stop"},

    {XK_Left, ControlMask | Mod1Mask, "skip", OXINE_KEY_SKIP, -30,
     "skip backward 30 secs"},
    {XK_Right, ControlMask | Mod1Mask, "skip", OXINE_KEY_SKIP, +30,
     "skip foreward 30 secs"},

    {XK_Left, ControlMask, "fast_rewind", OXINE_KEY_FASTRWD, 0,
     "fast rewind"},
    {XK_Right, ControlMask, "fast_forward", OXINE_KEY_FASTFWD, 0,
     "fast foreward"},

    {XK_Up, ControlMask, "speed", OXINE_KEY_SPEED, +1,
     "increase playback speed"},
    {XK_Down, ControlMask, "speed", OXINE_KEY_SPEED, -1,
     "decrease playback speed"},

    {XK_e, 0, "eject", OXINE_KEY_EJECT, 0, NULL},
    {XK_E, 0, "eject", OXINE_KEY_EJECT, 0, NULL},

    {XK_a, 0, "toggle_aspect_ratio", OXINE_KEY_ASPECT_RATIO, 0, NULL},
    {XK_A, 0, "toggle_aspect_ratio", OXINE_KEY_ASPECT_RATIO, 0, NULL},

    {XK_i, 0, "toggle_deinterlace", OXINE_KEY_DEINTERLACE, 0, NULL},
    {XK_I, 0, "toggle_deinterlace", OXINE_KEY_DEINTERLACE, 0, NULL},

    {0, 0, "saturation", OXINE_KEY_SATURATION, 500, NULL},
    {0, 0, "brightness", OXINE_KEY_BRIGHTNESS, 500, NULL},
    {0, 0, "contrast", OXINE_KEY_CONTRAST, 500, NULL},
    {0, 0, "hue", OXINE_KEY_HUE, 500, NULL},

    /* Navigation */
    {XK_o, 0, "osdmenu", OXINE_KEY_MENU_OSD, 0, NULL},
    {XK_O, 0, "osdmenu", OXINE_KEY_MENU_OSD, 0, NULL},

    {0, 0, "vdr", OXINE_KEY_PLAY_VDR, 0, "play VDR"},
    {0, 0, "dvb", OXINE_KEY_PLAY_DVB, 0, "play DVB"},
    {0, 0, "v4l", OXINE_KEY_PLAY_V4L, 0, "play V4L"},
    {XK_F9, 0, "playbackmenu", OXINE_KEY_MENU_CURRENT_TITLE, 0, "jump to playback menu"},
    {XK_F10, 0, "mainmenu", OXINE_KEY_MENU_MAIN, 0, "jump to main menu"},
    {XK_F11, 0, "musicmenu", OXINE_KEY_MENU_MUSIC, 0, "jump to music menu"},
    {XK_F12, 0, "videomenu", OXINE_KEY_MENU_VIDEO, 0, "jump to video menu"},

    {XK_Left, 0, "left", OXINE_KEY_LEFT, 0, NULL},
    {XK_Right, 0, "right", OXINE_KEY_RIGHT, 0, NULL},
    {XK_Up, 0, "up", OXINE_KEY_UP, 0, NULL},
    {XK_Down, 0, "down", OXINE_KEY_DOWN, 0, NULL},

    {XK_Home, 0, NULL, OXINE_KEY_FIRST, 0, NULL},
    {XK_End, 0, NULL, OXINE_KEY_LAST, 0, NULL},

    {XK_Page_Up, 0, NULL, OXINE_KEY_PAGE_UP, 0, NULL},
    {XK_Page_Down, 0, NULL, OXINE_KEY_PAGE_DOWN, 0, NULL},

    {XK_space, 0, "select", OXINE_KEY_SELECT, 0, NULL},

    {XK_Return, 0, "activate", OXINE_KEY_ACTIVATE, 0, NULL},
    {XK_Down, Mod1Mask, "activate", OXINE_KEY_ACTIVATE, 0, NULL},

    {0xFE20, 0, NULL, OXINE_KEY_PREV_WIDGET, 0, "prev widget"},
    {XK_Tab, 0, NULL, OXINE_KEY_NEXT_WIDGET, 0, "next widget"},

    {XK_p, 0, "prev", OXINE_KEY_PREV, 0, NULL},
    {XK_Left, Mod1Mask, "prev", OXINE_KEY_PREV, 0, NULL},
    {XK_Page_Up, Mod1Mask, "prev", OXINE_KEY_PREV, 0, NULL},
    {XK_n, 0, "next", OXINE_KEY_NEXT, 0, NULL},
    {XK_Right, Mod1Mask, "next", OXINE_KEY_NEXT, 0, NULL},
    {XK_Page_Down, Mod1Mask, "next", OXINE_KEY_NEXT, 0, NULL},

    {XK_Up, Mod1Mask, "back", OXINE_KEY_BACK, 0, NULL},
    {XK_Escape, 0, "back", OXINE_KEY_BACK, 0, NULL},
    {XK_BackSpace, 0, "back", OXINE_KEY_BACK, 0, NULL},

    {XK_Home, Mod1Mask, NULL, OXINE_KEY_HOME, 0, NULL},

    /* Misc */
    {XK_h, 0, "help", OXINE_KEY_HELP, 0, NULL},
    {XK_H, 0, "help", OXINE_KEY_HELP, 0, NULL},
    {XK_question, 0, "help", OXINE_KEY_HELP, 0, NULL},

    {XK_f, 0, NULL, OXINE_KEY_FULLSCREEN, 0, NULL},
    {XK_F, 0, NULL, OXINE_KEY_FULLSCREEN, 0, NULL},

    {XK_M, 0, "playmode", OXINE_KEY_PLAYMODE, 0, NULL},

    {XK_q, 0, "shutdown", OXINE_KEY_SHUTDOWN, 0, NULL},
    {XK_Q, 0, NULL, OXINE_KEY_QUIT, 0, NULL},

    {XK_Delete, 0, NULL, OXINE_KEY_REMOVE, 0, "remove"},

    /* VDR */
#ifdef HAVE_VDR
    {0, 0, "record", OXINE_KEY_RECORD, 0, NULL},
    /* RED, GREEN, YELLOW are mapped to MENU5, MENU6 and MENU7 in VDR mode. */
    {0, 0, "red", OXINE_KEY_RED, 0, NULL},
    {0, 0, "green", OXINE_KEY_GREEN, 0, NULL},
    {0, 0, "yellow", OXINE_KEY_YELLOW, 0, NULL},
    {XK_F8, 0, "blue", OXINE_KEY_BLUE, 0, NULL},
#endif

    /* The End */
    {0, 0, 0, 0, 0, NULL}
};

int
odk_keymap_kbd2action (oxine_event_t * event, KeySym kbd_key,
                       int kbd_modifier)
{
    odk_keymap_entry_t *entry = &default_keymap_table[0];

#ifdef DEBUG_MODIFIERS
    if (kbd_modifier & ControlMask)
        debug ("detected modifier: CTRL");
    if (kbd_modifier & Mod1Mask)
        debug ("detected modifier: MOD1");
    if (kbd_modifier & Mod2Mask)
        debug ("detected modifier: MOD2");
    if (kbd_modifier & Mod3Mask)
        debug ("detected modifier: MOD3");
#endif

    /* We only care about Control (Control_L) and Mod1 (Meta_L Meta_R) */
    kbd_modifier &= ControlMask | Mod1Mask;

    event->source.key = OXINE_KEY_NULL;
    event->data.how = 0;
    while (entry->key_id != 0) {
        if ((entry->kbd_key == kbd_key)
            && ((!entry->kbd_modifier && !kbd_modifier)
                || (entry->kbd_modifier == kbd_modifier))) {
            event->source.key = entry->key_id;
            event->data.how = entry->how;
#ifdef DEBUG
            if (entry->description)
                debug ("command: %s", entry->description);
            else if (entry->lirc_action)
                debug ("command: %s", entry->lirc_action);
#endif
            return 1;
        }
        entry++;
    }

    return 0;
}


int
odk_keymap_lirc2action (oxine_event_t * event, char *lirc_action)
{
    odk_keymap_entry_t *entry = &default_keymap_table[0];

    event->source.key = OXINE_KEY_NULL;
    event->data.how = 0;
    while (entry->key_id != 0) {
        if ((entry->lirc_action)
            && (strncasecmp (entry->lirc_action, lirc_action,
                             strlen (entry->lirc_action)) == 0)) {
            event->source.key = entry->key_id;
            event->data.how = entry->how;
#ifdef DEBUG
            if (entry->description)
                debug ("command: %s", entry->description);
            else if (entry->lirc_action)
                debug ("command: %s", entry->lirc_action);
#endif
            return 1;
        }
        entry++;
    }

    return 0;
}
