/* -*- Mode: C; c-basic-offset: 2; indent-tabs-mode: nil -*-
 *
 * Pigment OpenGL ES-CM 1.1 plugin
 *
 * Copyright © 2008 Fluendo Embedded S.L.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author(s): Loïc Molinari <loic@fluendo.com>
 *            Mirco Müller <macslow@bangang.de>
 */

/*
 * PgmGlesText uses the Pango text rendering pipeline to generate a pixmap with
 * the size of the drawable and then bind it as a texture. Each time a property
 * is changed, the pixmap is updated. When the drawable size is changed, the
 * pixmap is reallocated to this new one and then updated.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */

#include <math.h>
#include "pgmglestext.h"

#ifndef HAVE_PANGO_1_16
#ifndef G_OS_WIN32
#include <locale.h>
#endif
#endif

/* This is the offset such that (0xFF << PGM_n_OFFSET) is the mask to access the
 * nth byte of a guint32 */
#if G_BYTE_ORDER == G_LITTLE_ENDIAN
#define BYTE_1_OFFSET 0
#define BYTE_2_OFFSET 8
#define BYTE_3_OFFSET 16
#define BYTE_4_OFFSET 24
#else /* G_BIG_ENDIAN */
#define BYTE_1_OFFSET 24
#define BYTE_2_OFFSET 16
#define BYTE_3_OFFSET 8
#define BYTE_4_OFFSET 0
#endif

/* These are the offsets to shift to access the alpha, red, green and blue bytes
 * in a native endian ARGB value (such as what is given by Cairo) */
#define ARGB_A_OFFSET 24
#define ARGB_R_OFFSET 16
#define ARGB_G_OFFSET 8
#define ARGB_B_OFFSET 0

/* These are the A,R,G,B component byte masks for a native endian ARGB value
 * (such as what is given by Cairo) */
#define ARGB_A_MASK 0xFF << ARGB_A_OFFSET
#define ARGB_R_MASK 0xFF << ARGB_R_OFFSET
#define ARGB_G_MASK 0xFF << ARGB_G_OFFSET
#define ARGB_B_MASK 0xFF << ARGB_B_OFFSET

GST_DEBUG_CATEGORY_STATIC (pgm_gles_text_debug);
#define GST_CAT_DEFAULT pgm_gles_text_debug

static PgmGlesDrawableClass *parent_class = NULL;

/* Private methods */

/* Gets the projected height of the drawable on the viewport */
static gfloat
get_projected_height (PgmGlesDrawable *glesdrawable,
                      gfloat canvas_height)
{
  PgmGlesViewport *glesviewport = glesdrawable->glesviewport;
  PgmViewport *viewport = PGM_VIEWPORT (glesviewport);
  PgmCanvas *canvas = viewport->canvas;
  gfloat height;

  /* FIXME: That's just the projected height on z=0 */
  GST_OBJECT_LOCK (canvas);
  GST_OBJECT_LOCK (viewport);

  if (viewport->rotation == PGM_VIEWPORT_ROTATION_NONE
      || viewport->rotation == PGM_VIEWPORT_ROTATION_180)
    height = (canvas_height * viewport->projected_height) / canvas->height;
  else
    height = (canvas_height * viewport->projected_height) / canvas->width;

  GST_OBJECT_UNLOCK (viewport);
  GST_OBJECT_UNLOCK (canvas);

  return height;
}

/* Gets the projected width of the drawable on the viewport */
static gfloat
get_projected_width (PgmGlesDrawable *glesdrawable,
                     gfloat canvas_width)
{
  PgmGlesViewport *glesviewport = glesdrawable->glesviewport;
  PgmViewport *viewport = PGM_VIEWPORT (glesviewport);
  PgmCanvas *canvas = viewport->canvas;
  gfloat width;

  /* FIXME: That's just the projected width on z=0 */
  GST_OBJECT_LOCK (canvas);
  GST_OBJECT_LOCK (viewport);

  if (viewport->rotation == PGM_VIEWPORT_ROTATION_NONE
      || viewport->rotation == PGM_VIEWPORT_ROTATION_180)
    width = (canvas_width * viewport->projected_width) / canvas->width;
  else
    width = (canvas_width * viewport->projected_width) / canvas->height;

  GST_OBJECT_UNLOCK (viewport);
  GST_OBJECT_UNLOCK (canvas);

  return width;
}

/* Defines texture coordinates */
static void
set_coordinates (PgmGlesText *glestext)
{
  PgmGlesTexture *glestexture = glestext->texture;

  glestext->coord[2] = (PgmGlesFloat) glestexture->width
    / glestexture->width_pot;
  glestext->coord[4] = glestext->coord[2];
  glestext->coord[5] = (PgmGlesFloat) glestexture->height
    / glestexture->height_pot;
  glestext->coord[7] = glestext->coord[5];
}

/* Stores outline width depending on PgmText outline width converting it
 * from canvas coordinates to viewport coordinates */
static void
set_outline_width (PgmGlesText *glestext)
{
  PgmGlesDrawable *glesdrawable = PGM_GLES_DRAWABLE (glestext);
  PgmText *text = PGM_TEXT (glesdrawable->drawable);

  glestext->outline_width = get_projected_height (glesdrawable,
                                                  text->outline_width);
}

/* Stores outline color depending on PgmText color normalizing components to
 * the range [0.0; 1.0] */
static void
set_outline_color (PgmGlesText *glestext)
{
  PgmGlesDrawable *glesdrawable = PGM_GLES_DRAWABLE (glestext);
  PgmText *text = PGM_TEXT (glesdrawable->drawable);

  GST_OBJECT_LOCK (text);
  glestext->outline_color[0] = text->outline_r * INV_255;
  glestext->outline_color[1] = text->outline_g * INV_255;
  glestext->outline_color[2] = text->outline_b * INV_255;
  glestext->outline_color[3] = text->outline_a * INV_255;
  GST_OBJECT_UNLOCK (text);
}

/* Defines Pango spacing depending on PgmText line spacing converting it
 * from canvas coordinates to viewport coordinates */
static void
set_line_spacing (PgmGlesText *glestext)
{
  PgmGlesDrawable *glesdrawable = PGM_GLES_DRAWABLE (glestext);
  PgmText *text = PGM_TEXT (glesdrawable->drawable);
  gfloat line_spacing;

  line_spacing = get_projected_height (glesdrawable, text->line_spacing);
  pango_layout_set_spacing (glestext->layout, (gint)
                            (line_spacing * PANGO_SCALE));
}

/* Calculates the maximum total height of a line of text for the current font
 * and font height set in glestext. Returns the value in device unit. */
static gfloat
get_max_height (PgmGlesText *glestext)
{
  /* FIXME: we use the language of the current locale, but the text we display
   *        may be in another language. */
  PangoLanguage *language = NULL;
  PangoFontMetrics *metrics = NULL;
  gint ascent, descent;
  gchar *locale = NULL;

#ifdef HAVE_PANGO_1_16
  language = pango_language_get_default ();
#else
#ifdef G_OS_WIN32
  locale = g_win32_getlocale ();
#else
  locale = g_strdup (setlocale (LC_CTYPE, NULL));
#endif
  language = pango_language_from_string (locale);
#endif

  metrics = pango_context_get_metrics (glestext->pango_ctx,
                                       glestext->desc, language);
  ascent = pango_font_metrics_get_ascent (metrics);
  descent = pango_font_metrics_get_descent (metrics);

  pango_font_metrics_unref (metrics);

  g_free (locale);

  return ((ascent + descent) / ((gfloat) PANGO_SCALE));
}

/* Defines Pango font size depending on PgmText height converting it from
 * canvas coordinates to viewport coordinates. This function depends on
 * various parameters, including the font set for glestext. When setting many
 * parameters, it is cautious to set this one last. */
static void
set_font_height (PgmGlesText *glestext)
{
  PgmGlesDrawable *glesdrawable = PGM_GLES_DRAWABLE (glestext);
  PgmText *text = PGM_TEXT (glesdrawable->drawable);

  /* All the heights here are in device unit. */
  gfloat requested_height, wanted_height, max_height, font_ratio;

  requested_height = get_projected_height (glesdrawable, text->height);
  requested_height *= glesdrawable->height;

  /* The "pango height" is something like the height of an M, but we want to set
   * the maximum height of a line, so we need to calculate the ratio first. */
  pango_font_description_set_absolute_size (glestext->desc,
                                            requested_height * PANGO_SCALE);
  max_height = get_max_height (glestext);
  font_ratio = requested_height / max_height;

  wanted_height = requested_height * font_ratio;
  pango_font_description_set_absolute_size (glestext->desc,
                                            wanted_height * PANGO_SCALE);
  pango_layout_set_font_description (glestext->layout, glestext->desc);
}

/* Defines Pango font family depending on PgmText font family */
static void
set_font_family (PgmGlesText *glestext)
{
  PgmGlesDrawable *glesdrawable = PGM_GLES_DRAWABLE (glestext);
  PgmText *text = PGM_TEXT (glesdrawable->drawable);

  GST_OBJECT_LOCK (text);
  pango_font_description_set_family (glestext->desc, text->font_family);
  pango_layout_set_font_description (glestext->layout, glestext->desc);
  GST_OBJECT_UNLOCK (text);
}

/* Defines Pango text/markup depending on PgmText label */
static void
set_label (PgmGlesText *glestext)
{
  PgmGlesDrawable *glesdrawable = PGM_GLES_DRAWABLE (glestext);
  PgmText *text = PGM_TEXT (glesdrawable->drawable);

  /* FIXME: defining a markup and right after a label keeps the
   *        previous markups on the new label. */
  GST_OBJECT_LOCK (text);
  if (text->use_markup)
    pango_layout_set_markup (glestext->layout, text->label, -1);
  else
    pango_layout_set_text (glestext->layout, text->label, -1);
  GST_OBJECT_UNLOCK (text);
}

/* Defines Pango width depending on PgmText width */
static void
set_width (PgmGlesText *glestext)
{
  PgmGlesDrawable *glesdrawable = PGM_GLES_DRAWABLE (glestext);
  PgmViewport *viewport = PGM_VIEWPORT (glesdrawable->glesviewport);
  gfloat non_square_ratio;
  gfloat width;

  width = get_projected_width (glesdrawable, glesdrawable->width);

  GST_OBJECT_LOCK (viewport);
  non_square_ratio = 1.0f / viewport->pixel_aspect_ratio;
  GST_OBJECT_UNLOCK (viewport);

  pango_layout_set_width (glestext->layout, (guint)
                          (width * PANGO_SCALE * non_square_ratio));
}

/* Defines Pango justification depending on PgmText justification */
static void
set_justify (PgmGlesText *glestext)
{
#ifdef HAVE_PANGO_1_16
  PgmGlesDrawable *glesdrawable = PGM_GLES_DRAWABLE (glestext);
  PgmText *text = PGM_TEXT (glesdrawable->drawable);

  GST_OBJECT_LOCK (text);
  pango_layout_set_justify (glestext->layout, text->justify);
  GST_OBJECT_UNLOCK (text);
#endif /* HAVE_PANGO_1_16 */
}

/* Defines Pango weight depending on PgmText weight */
static void
set_weight (PgmGlesText *glestext)
{
  PgmGlesDrawable *glesdrawable = PGM_GLES_DRAWABLE (glestext);
  PgmText *text = PGM_TEXT (glesdrawable->drawable);

  switch (text->weight)
    {
    case PGM_TEXT_WEIGHT_LIGHT:
      pango_font_description_set_weight (glestext->desc, PANGO_WEIGHT_LIGHT);
      break;

    case PGM_TEXT_WEIGHT_NORMAL:
      pango_font_description_set_weight (glestext->desc, PANGO_WEIGHT_NORMAL);
      break;

    case PGM_TEXT_WEIGHT_BOLD:
      pango_font_description_set_weight (glestext->desc, PANGO_WEIGHT_BOLD);
      break;

    default:
      break;
    }

  pango_layout_set_font_description (glestext->layout, glestext->desc);
}

/* Defines Pango stretching depending on PgmText stretching */
static void
set_stretch (PgmGlesText *glestext)
{
  PgmGlesDrawable *glesdrawable = PGM_GLES_DRAWABLE (glestext);
  PgmText *text = PGM_TEXT (glesdrawable->drawable);

  switch (text->stretch)
    {
    case PGM_TEXT_STRETCH_CONDENSED:
      pango_font_description_set_stretch (glestext->desc,
                                          PANGO_STRETCH_CONDENSED);
      break;

    case PGM_TEXT_STRETCH_NORMAL:
      pango_font_description_set_stretch (glestext->desc, PANGO_STRETCH_NORMAL);
      break;

    case PGM_TEXT_STRETCH_EXPANDED:
      pango_font_description_set_stretch (glestext->desc,
                                          PANGO_STRETCH_EXPANDED);
      break;

    default:
      break;
    }

  pango_layout_set_font_description (glestext->layout, glestext->desc);
}

/* Defines Pango alignment depending on PgmText alignment */
static void
set_alignment (PgmGlesText *glestext)
{
  PgmGlesDrawable *glesdrawable = PGM_GLES_DRAWABLE (glestext);
  PgmText *text = PGM_TEXT (glesdrawable->drawable);

  switch (text->alignment)
    {
    case PGM_TEXT_ALIGN_LEFT:
      pango_layout_set_alignment (glestext->layout, PANGO_ALIGN_LEFT);
      break;

    case PGM_TEXT_ALIGN_CENTER:
      pango_layout_set_alignment (glestext->layout, PANGO_ALIGN_CENTER);
      break;

    case PGM_TEXT_ALIGN_RIGHT:
      pango_layout_set_alignment (glestext->layout, PANGO_ALIGN_RIGHT);
      break;

    default:
      break;
    }
}

/* Defines Pango ellipsizing depending on PgmText ellipsizing */
static void
set_ellipsize (PgmGlesText *glestext)
{
  PgmGlesDrawable *glesdrawable = PGM_GLES_DRAWABLE (glestext);
  PgmText *text = PGM_TEXT (glesdrawable->drawable);

  switch (text->ellipsize)
    {
    case PGM_TEXT_ELLIPSIZE_NONE:
      pango_layout_set_ellipsize (glestext->layout, PANGO_ELLIPSIZE_NONE);
      break;

    case PGM_TEXT_ELLIPSIZE_START:
      pango_layout_set_ellipsize (glestext->layout, PANGO_ELLIPSIZE_START);
      break;

    case PGM_TEXT_ELLIPSIZE_MIDDLE:
      pango_layout_set_ellipsize (glestext->layout, PANGO_ELLIPSIZE_MIDDLE);
      break;

    case PGM_TEXT_ELLIPSIZE_END:
      pango_layout_set_ellipsize (glestext->layout, PANGO_ELLIPSIZE_END);
      break;

    default:
      break;
    }
}

/* Defines Pango wrapping depending on PgmText wrapping */
static void
set_wrap (PgmGlesText *glestext)
{
  PgmGlesDrawable *glesdrawable = PGM_GLES_DRAWABLE (glestext);
  PgmText *text = PGM_TEXT (glesdrawable->drawable);

  switch (text->wrap)
    {
    case PGM_TEXT_WRAP_WORD:
      pango_layout_set_wrap (glestext->layout, PANGO_WRAP_WORD);
      break;

    case PGM_TEXT_WRAP_CHAR:
      pango_layout_set_wrap (glestext->layout, PANGO_WRAP_CHAR);
      break;

    case PGM_TEXT_WRAP_WORD_CHAR:
      pango_layout_set_wrap (glestext->layout, PANGO_WRAP_WORD_CHAR);
      break;

    default:
      break;
    }
}

/* Defines Pango gravity depending on PgmText gravity */
static void
set_gravity (PgmGlesText *glestext)
{
#ifdef HAVE_PANGO_1_16
  PgmGlesDrawable *glesdrawable = PGM_GLES_DRAWABLE (glestext);
  PgmText *text = PGM_TEXT (glesdrawable->drawable);

  switch (text->gravity)
    {
    case PGM_TEXT_GRAVITY_SOUTH:
      pango_context_set_base_gravity (glestext->pango_ctx, PANGO_GRAVITY_SOUTH);
      break;

    case PGM_TEXT_GRAVITY_EAST:
      pango_context_set_base_gravity (glestext->pango_ctx, PANGO_GRAVITY_EAST);
      break;

    case PGM_TEXT_GRAVITY_NORTH:
      pango_context_set_base_gravity (glestext->pango_ctx, PANGO_GRAVITY_NORTH);
      break;

    case PGM_TEXT_GRAVITY_WEST:
      pango_context_set_base_gravity (glestext->pango_ctx, PANGO_GRAVITY_WEST);
      break;

    case PGM_TEXT_GRAVITY_AUTO:
      pango_context_set_base_gravity (glestext->pango_ctx, PANGO_GRAVITY_AUTO);
      break;

    default:
      break;
    }
#endif /* HAVE_PANGO_1_16 */
}

/* Defines Pango style depending on PgmText style */
static void
set_style (PgmGlesText *glestext)
{
  PgmGlesDrawable *glesdrawable = PGM_GLES_DRAWABLE (glestext);
  PgmText *text = PGM_TEXT (glesdrawable->drawable);

  switch (text->style)
    {
    case PGM_TEXT_STYLE_NORMAL:
      pango_font_description_set_style (glestext->desc, PANGO_STYLE_NORMAL);
      break;

    case PGM_TEXT_STYLE_OBLIQUE:
      pango_font_description_set_style (glestext->desc, PANGO_STYLE_OBLIQUE);
      break;

    case PGM_TEXT_STYLE_ITALIC:
      pango_font_description_set_style (glestext->desc, PANGO_STYLE_ITALIC);
      break;

    default:
      break;
    }

  pango_layout_set_font_description (glestext->layout, glestext->desc);
}

/* Defines Pango variant depending on PgmText variant */
static void
set_variant (PgmGlesText *glestext)
{
  PgmGlesDrawable *glesdrawable = PGM_GLES_DRAWABLE (glestext);
  PgmText *text = PGM_TEXT (glesdrawable->drawable);

  switch (text->variant)
    {
    case PGM_TEXT_VARIANT_NORMAL:
      pango_font_description_set_variant (glestext->desc, PANGO_VARIANT_NORMAL);
      break;

    case PGM_TEXT_VARIANT_SMALL_CAPS:
      pango_font_description_set_variant (glestext->desc,
                                          PANGO_VARIANT_SMALL_CAPS);
      break;

    default:
      break;
    }

  pango_layout_set_font_description (glestext->layout, glestext->desc);
}

/* Update all the properties of the text */
static void
update_properties (PgmGlesText *glestext)
{
  set_font_family (glestext);
  set_weight (glestext);
  set_stretch (glestext);
  set_style (glestext);
  set_variant (glestext);
  set_alignment (glestext);
  set_ellipsize (glestext);
  set_justify (glestext);
  set_line_spacing (glestext);
  set_width (glestext);
  set_wrap (glestext);
  set_gravity (glestext);
  set_label (glestext);
  set_outline_width (glestext);
  set_outline_color (glestext);
  set_font_height (glestext);
}

/* Cairo uses premultiplied-alpha to represent pixels (mostly for compositing).
 * This function unpremultiplies the alpha of cairo's native endian ARGB buffer
 * and converts it to BGRA in memory order. */
static void
unpremultiply_alpha (guint32 *buffer,
                     guint width,
                     guint height)
{
  guint size = width * height;
  guint8 a, r, g, b;
  guint i;

  for (i = 0; i < size; i++)
    {
      a = (buffer[i] & ARGB_A_MASK) >> ARGB_A_OFFSET;
      if (a == 0) /* optimisation for most common case */
        {
          continue;
        }
      else if (a == 255)
        {
          /* If the pixel is full alpha, we don't have to change the rgb values,
           * but we have to convert the pixel from native endian ARGB to memory
           * BGRA, which is equivalent to little endian ARGB. */
          buffer[i] = GUINT32_TO_LE (buffer[i]);
        }
      else
        {
          r = (((buffer[i] & ARGB_R_MASK) >> ARGB_R_OFFSET)
               * 255 + a / 2) / a;
          g = (((buffer[i] & ARGB_G_MASK) >> ARGB_G_OFFSET)
               * 255 + a / 2) / a;
          b = (((buffer[i] & ARGB_B_MASK) >> ARGB_B_OFFSET)
               * 255 + a / 2) / a;
          /* we write the buffer in BGRA memory order (B in low address, A in
           * high address) in one memory access */
          buffer[i] = b << BYTE_1_OFFSET
              | g << BYTE_2_OFFSET
              | r << BYTE_3_OFFSET
              | a << BYTE_4_OFFSET;
        }
    }
}

/* Allocates the pixmap, the Pango pipeline using a Cairo backend and
 * generates the texture */
static void
create_pixmap (PgmGlesText *glestext)
{
  PgmGlesDrawable *glesdrawable = PGM_GLES_DRAWABLE (glestext);
  PgmViewport *viewport = PGM_VIEWPORT (glesdrawable->glesviewport);
  PgmGlesContextTask *task;
  gfloat pixel_aspect_ratio;

  glestext->width = (guint) get_projected_width (glesdrawable,
                                                 glesdrawable->width);
  glestext->height = (guint) get_projected_height (glesdrawable,
                                                   glesdrawable->height);

  GST_OBJECT_LOCK (viewport);
  pixel_aspect_ratio = viewport->pixel_aspect_ratio;
  GST_OBJECT_UNLOCK (viewport);

  glestext->size = glestext->width * glestext->height * 4;
  glestext->buffer = (guchar *) g_slice_alloc0 (glestext->size);
  glestext->surface = cairo_image_surface_create_for_data (glestext->buffer,
                                                           CAIRO_FORMAT_ARGB32,
                                                           glestext->width,
                                                           glestext->height,
                                                           glestext->width * 4);

  glestext->cairo_ctx = cairo_create (glestext->surface);
  cairo_set_font_options (glestext->cairo_ctx, glestext->font_options);
  cairo_scale (glestext->cairo_ctx, pixel_aspect_ratio, 1.0f);
  glestext->layout = pango_cairo_create_layout (glestext->cairo_ctx);
  glestext->pango_ctx = pango_layout_get_context (glestext->layout);

  pgm_gles_texture_set_buffer (glestext->texture, glestext->buffer,
                               PGM_IMAGE_RGBA, glestext->width, glestext->height,
                               glestext->size, 0);
  set_coordinates (glestext);

  /* And request a texture generation */
  task = pgm_gles_context_task_new (PGM_GLES_CONTEXT_GEN_TEXTURE,
                                    glestext->texture);
  pgm_gles_context_push_immediate_task (glesdrawable->glesviewport->context,
                                        task);
}

/* Frees the pixmap and the Cairo backend */
static void
free_pixmap (PgmGlesText *glestext)
{
  PgmGlesDrawable *glesdrawable = PGM_GLES_DRAWABLE (glestext);
  PgmGlesContextTask *task;

  g_object_unref (glestext->layout);
  glestext->layout = NULL;
  cairo_destroy (glestext->cairo_ctx);
  glestext->cairo_ctx = NULL;
  cairo_surface_destroy (glestext->surface);
  glestext->surface = NULL;
  if (glestext->buffer)
    {
      g_slice_free1 (glestext->size, (gpointer) glestext->buffer);
      glestext->buffer = NULL;
    }

  /* Request texture clean up */
  task = pgm_gles_context_task_new (PGM_GLES_CONTEXT_CLEAN_TEXTURE,
                                    glestext->texture);
  pgm_gles_context_push_immediate_task (glesdrawable->glesviewport->context,
                                        task);
}

/* Updates text properties and draw it on the pixmap */
static void
update_pixmap (PgmGlesText *glestext)
{
  PgmGlesDrawable *glesdrawable = PGM_GLES_DRAWABLE (glestext);
  PgmGlesContextTask *task;

  /* Make context fully transparent */
  cairo_set_operator (glestext->cairo_ctx, CAIRO_OPERATOR_SOURCE);
  cairo_set_source_rgba (glestext->cairo_ctx, 0.0f, 0.0f, 0.0f, 0.0f);
  cairo_paint (glestext->cairo_ctx);

  /* Then draw the text */
  cairo_set_source_rgba (glestext->cairo_ctx, 1.0f, 1.0f, 1.0f, 1.0f);
  pango_cairo_show_layout (glestext->cairo_ctx, glestext->layout);

  /* And the stroke */
  if (glestext->outline_width > 0.0f)
    {
      pango_cairo_layout_path (glestext->cairo_ctx, glestext->layout);
      cairo_set_source_rgba (glestext->cairo_ctx,
                             glestext->outline_color[0],
                             glestext->outline_color[1],
                             glestext->outline_color[2],
                             glestext->outline_color[3]);
      cairo_set_line_width (glestext->cairo_ctx, glestext->outline_width);
      cairo_stroke (glestext->cairo_ctx);
    }

#if 0
  /* Debugging */
#ifndef WIN32
  cairo_surface_write_to_png (glestext->surface, "/tmp/pgmglestext.png");
#else
  cairo_surface_write_to_png (glestext->surface, "C:\\pgmglestext.png");
#endif /* WIN32 */
#endif

  /* Get rid of the premultiplied alpha */
  unpremultiply_alpha ((guint32*) glestext->buffer, glestext->width,
                       glestext->height);

  /* Update texture */
 pgm_gles_texture_set_buffer (glestext->texture, glestext->buffer,
                              PGM_IMAGE_BGRA, glestext->width, glestext->height,
                              glestext->size, 0);

  /* Request texture upload */
  task = pgm_gles_context_task_new (PGM_GLES_CONTEXT_UPLOAD_TEXTURE,
                                    glestext->texture);
  pgm_gles_context_push_deferred_task (glesdrawable->glesviewport->context,
                                       task);
}

/* Do a complete regeneration of a text */
static void
regenerate_text (PgmGlesText *glestext)
{
  free_pixmap (glestext);
  create_pixmap (glestext);
  update_properties (glestext);
  update_pixmap (glestext);
}

/* Update a text by simply generating the new one in the current allocated
 * memory if the size has not changed. If it has changed, the whole text is
 * regenerated. */
static void
update_text (PgmGlesText *glestext)
{
  if (!glestext->size_updated)
    update_pixmap (glestext);
  else
    {
      glestext->size_updated = FALSE;
      regenerate_text (glestext);
    }
}

/* PgmGlesDrawable methods */

static void
pgm_gles_text_draw (PgmGlesDrawable *glesdrawable)
{
  static PgmGlesUshort indices[] = { 0, 1, 2, 2, 3, 0 };
  PgmGlesText *glestext = PGM_GLES_TEXT (glesdrawable);

  GST_LOG_OBJECT (glesdrawable, "draw");

  /* Only draw if it's visible */
  if (glestext->fg_color[3] != 0.0f)
    {
      PgmGlesContextProcAddress *gles =
        glesdrawable->glesviewport->context->gles;

      pgm_gles_texture_bind (glestext->texture);
      gles->enable_client_state (PGM_GLES_VERTEX_ARRAY);
      gles->enable_client_state (PGM_GLES_COLOR_ARRAY);
      gles->enable_client_state (PGM_GLES_TEXTURE_COORD_ARRAY);
      gles->vertex_pointer (3, PGM_GLES_FLOAT, 0, glestext->vertex);
      gles->color_pointer (4, PGM_GLES_FLOAT, 0, glestext->fg_color);
      gles->tex_coord_pointer (2, PGM_GLES_FLOAT, 0, glestext->coord);
      gles->draw_elements (PGM_GLES_TRIANGLES, 6,
                           PGM_GLES_UNSIGNED_SHORT, indices);
      gles->disable_client_state (PGM_GLES_VERTEX_ARRAY);
      gles->disable_client_state (PGM_GLES_COLOR_ARRAY);
      gles->disable_client_state (PGM_GLES_TEXTURE_COORD_ARRAY);
      pgm_gles_texture_unbind (glestext->texture);
    }
}

static void
pgm_gles_text_regenerate (PgmGlesDrawable *glesdrawable)
{
  PgmGlesText *glestext = PGM_GLES_TEXT (glesdrawable);

  GST_DEBUG_OBJECT (glesdrawable, "regenerate");

  if (glestext->size_updated)
    {
      regenerate_text (glestext);
      glestext->size_updated = FALSE;
    }
}

static void
pgm_gles_text_update_projection (PgmGlesDrawable *glesdrawable)
{
  PgmGlesText *glestext = PGM_GLES_TEXT (glesdrawable);

  GST_DEBUG_OBJECT (glesdrawable, "update_projection");

  glestext->size_updated = TRUE;
}

static void
pgm_gles_text_set_size (PgmGlesDrawable *glesdrawable)
{
  PgmGlesText *glestext = PGM_GLES_TEXT (glesdrawable);
  guint i;

  GST_DEBUG_OBJECT (glesdrawable, "set_size");

  for (i = 0; i < 12; i++)
    glestext->vertex[i] = glesdrawable->bg_vertex[i];

  glestext->size_updated = TRUE;
}

static void
pgm_gles_text_set_position (PgmGlesDrawable *glesdrawable)
{
  PgmGlesText *glestext = PGM_GLES_TEXT (glesdrawable);
  guint i;

  GST_DEBUG_OBJECT (glesdrawable, "set_position");

  for (i = 0; i < 12; i++)
    glestext->vertex[i] = glesdrawable->bg_vertex[i];
}

static void
pgm_gles_text_set_fg_color (PgmGlesDrawable *glesdrawable)
{
  PgmGlesText *glestext = PGM_GLES_TEXT (glesdrawable);
  PgmDrawable *drawable;
  PgmGlesFloat color[4];

  GST_DEBUG_OBJECT (glesdrawable, "set_fg_color");

  drawable = glesdrawable->drawable;

  color[0] = drawable->fg_r * INV_255;
  color[1] = drawable->fg_g * INV_255;
  color[2] = drawable->fg_b * INV_255;
  color[3] = drawable->fg_a * drawable->opacity * INV_255;

  GST_OBJECT_LOCK (drawable);
  glestext->fg_color[0] = color[0];
  glestext->fg_color[1] = color[1];
  glestext->fg_color[2] = color[2];
  glestext->fg_color[3] = color[3];
  glestext->fg_color[4] = color[0];
  glestext->fg_color[5] = color[1];
  glestext->fg_color[6] = color[2];
  glestext->fg_color[7] = color[3];
  glestext->fg_color[8] = color[0];
  glestext->fg_color[9] = color[1];
  glestext->fg_color[10] = color[2];
  glestext->fg_color[11] = color[3];
  glestext->fg_color[12] = color[0];
  glestext->fg_color[13] = color[1];
  glestext->fg_color[14] = color[2];
  glestext->fg_color[15] = color[3];
  GST_OBJECT_UNLOCK (drawable);
}

static void
pgm_gles_text_set_opacity (PgmGlesDrawable *glesdrawable)
{
  PgmGlesText *glestext = PGM_GLES_TEXT (glesdrawable);
  PgmDrawable *drawable;
  PgmGlesFloat alpha;

  GST_DEBUG_OBJECT (glesdrawable, "set_opacity");

  drawable = glesdrawable->drawable;
  alpha = drawable->fg_a * drawable->opacity * SQR_INV_255;

  GST_OBJECT_LOCK (drawable);
  glestext->fg_color[3] = alpha;
  glestext->fg_color[7] = alpha;
  glestext->fg_color[11] = alpha;
  glestext->fg_color[15] = alpha;
  GST_OBJECT_UNLOCK (drawable);
}

static void
pgm_gles_text_sync (PgmGlesDrawable *glesdrawable)
{
  PgmGlesText *glestext = PGM_GLES_TEXT (glesdrawable);

  GST_DEBUG_OBJECT (glesdrawable, "sync");

  create_pixmap (glestext);
  update_properties (glestext);

  /* Synchronize various other properties */
  pgm_gles_text_set_position (glesdrawable);
  pgm_gles_text_set_fg_color (glesdrawable);

  update_pixmap (glestext);
}

/* GObject stuff */

PGM_DEFINE_DYNAMIC_TYPE (PgmGlesText, pgm_gles_text, PGM_TYPE_GLES_DRAWABLE);

void
pgm_gles_text_register (GTypeModule *module)
{
  pgm_gles_text_register_type (module);
}

static void
pgm_gles_text_dispose (GObject *object)
{
  PgmGlesDrawable *glesdrawable = PGM_GLES_DRAWABLE (object);
  PgmGlesText *glestext = PGM_GLES_TEXT (object);
  PgmGlesContextTask *task;

  GST_DEBUG_OBJECT (glestext, "dispose");

  free_pixmap (glestext);

  pango_font_description_free (glestext->desc);
  glestext->desc = NULL;

  cairo_font_options_destroy (glestext->font_options);
  glestext->font_options = NULL;

  gst_object_unref (glesdrawable->drawable);
  pgm_gles_context_remove_tasks_with_data (glesdrawable->glesviewport->context,
                                           glestext->texture);
  task = pgm_gles_context_task_new (PGM_GLES_CONTEXT_FREE_TEXTURE,
                                    glestext->texture);
  pgm_gles_context_push_immediate_task (glesdrawable->glesviewport->context,
                                        task);
  glestext->texture = NULL;

  GST_CALL_PARENT (G_OBJECT_CLASS, dispose, (object));
}

static void
pgm_gles_text_class_init (PgmGlesTextClass *klass)
{
  GObjectClass *gobject_class;
  PgmGlesDrawableClass *glesdrawable_class;

  GST_DEBUG_CATEGORY_INIT (pgm_gles_text_debug, "pgm_gles_text", 0,
                           "OpenGL ES plugin: PgmGlesText");

  parent_class = g_type_class_peek_parent (klass);

  gobject_class = G_OBJECT_CLASS (klass);
  glesdrawable_class = PGM_GLES_DRAWABLE_CLASS (klass);

  /* GObject virtual table */
  gobject_class->dispose = GST_DEBUG_FUNCPTR (pgm_gles_text_dispose);

  /* PgmGlesDrawable virtual table */
  glesdrawable_class->sync = GST_DEBUG_FUNCPTR (pgm_gles_text_sync);
  glesdrawable_class->draw = GST_DEBUG_FUNCPTR (pgm_gles_text_draw);
  glesdrawable_class->regenerate = GST_DEBUG_FUNCPTR (pgm_gles_text_regenerate);
  glesdrawable_class->update_projection =
    GST_DEBUG_FUNCPTR (pgm_gles_text_update_projection);
  glesdrawable_class->set_size = GST_DEBUG_FUNCPTR (pgm_gles_text_set_size);
  glesdrawable_class->set_position =
    GST_DEBUG_FUNCPTR (pgm_gles_text_set_position);
  glesdrawable_class->set_fg_color =
    GST_DEBUG_FUNCPTR (pgm_gles_text_set_fg_color);
  glesdrawable_class->set_opacity =
    GST_DEBUG_FUNCPTR (pgm_gles_text_set_opacity);
}

static void
pgm_gles_text_class_finalize (PgmGlesTextClass *klass)
{
  return;
}

static void
pgm_gles_text_init (PgmGlesText *glestext)
{
  GST_DEBUG_OBJECT (glestext, "init");

  glestext->font_options = cairo_font_options_create ();
  cairo_font_options_set_antialias (glestext->font_options,
                                    CAIRO_ANTIALIAS_SUBPIXEL);

  glestext->desc = pango_font_description_new ();

  glestext->size_updated = FALSE;
}

/* Public methods */

PgmGlesDrawable *
pgm_gles_text_new (PgmDrawable *drawable,
                   PgmGlesViewport *glesviewport)
{
  PgmGlesDrawable *glesdrawable;
  PgmGlesText *glestext;

  glestext = g_object_new (PGM_TYPE_GLES_TEXT, NULL);

  GST_DEBUG_OBJECT (glestext, "created new glestext");

  glestext->texture = pgm_gles_texture_new (glesviewport->context);

  glesdrawable = PGM_GLES_DRAWABLE (glestext);
  glesdrawable->drawable = gst_object_ref (drawable);
  glesdrawable->glesviewport = glesviewport;
  pgm_gles_viewport_connect_changed_callback (glesviewport, glesdrawable);
  pgm_gles_drawable_sync (glesdrawable);

  return glesdrawable;
}

void
pgm_gles_text_set_label (PgmGlesText *glestext)
{
  GST_DEBUG_OBJECT (glestext, "set_label");

  set_label (glestext);
  update_text (glestext);
}

void
pgm_gles_text_set_markup (PgmGlesText *glestext)
{
  GST_DEBUG_OBJECT (glestext, "set_markup");

  set_label (glestext);
  update_text (glestext);
}

void
pgm_gles_text_set_font_family (PgmGlesText *glestext)
{
  GST_DEBUG_OBJECT (glestext, "set_font_family");

  set_font_family (glestext);
  update_text (glestext);
}

void
pgm_gles_text_set_font_height (PgmGlesText *glestext)
{
  GST_DEBUG_OBJECT (glestext, "set_font_height");

  set_font_height (glestext);
  update_text (glestext);
}

void
pgm_gles_text_set_ellipsize (PgmGlesText *glestext)
{
  GST_DEBUG_OBJECT (glestext, "set_ellipsize");

  set_ellipsize (glestext);
  update_text (glestext);
}

void
pgm_gles_text_set_justify (PgmGlesText *glestext)
{
  GST_DEBUG_OBJECT (glestext, "set_justify");

  set_justify (glestext);
  update_text (glestext);
}

void
pgm_gles_text_set_alignment (PgmGlesText *glestext)
{
  GST_DEBUG_OBJECT (glestext, "set_alignment");

  set_alignment (glestext);
  update_text (glestext);
}

void
pgm_gles_text_set_wrap (PgmGlesText *glestext)
{
  GST_DEBUG_OBJECT (glestext, "set_wrap");

  set_wrap (glestext);
  update_text (glestext);
}

void
pgm_gles_text_set_gravity (PgmGlesText *glestext)
{
  GST_DEBUG_OBJECT (glestext, "set_gravity");

  set_gravity (glestext);
  update_text (glestext);
}

void
pgm_gles_text_set_stretch (PgmGlesText *glestext)
{
  GST_DEBUG_OBJECT (glestext, "set_stretch");

  set_stretch (glestext);
  update_text (glestext);
}

void
pgm_gles_text_set_style (PgmGlesText *glestext)
{
  GST_DEBUG_OBJECT (glestext, "set_style");

  set_style (glestext);
  update_text (glestext);
}

void
pgm_gles_text_set_variant (PgmGlesText *glestext)
{
  GST_DEBUG_OBJECT (glestext, "set_variant");

  set_variant (glestext);
  update_text (glestext);
}

void
pgm_gles_text_set_weight (PgmGlesText *glestext)
{
  GST_DEBUG_OBJECT (glestext, "set_weight");

  set_weight (glestext);
  update_text (glestext);
}

void
pgm_gles_text_set_line_spacing (PgmGlesText *glestext)
{
  GST_DEBUG_OBJECT (glestext, "set_line_spacing");

  set_line_spacing (glestext);
  update_text (glestext);
}

void
pgm_gles_text_set_outline_color (PgmGlesText *glestext)
{
  GST_DEBUG_OBJECT (glestext, "set_outline_color");

  set_outline_color (glestext);
  update_text (glestext);
}

void
pgm_gles_text_set_outline_width (PgmGlesText *glestext)
{
  GST_DEBUG_OBJECT (glestext, "set_outline_width");

  set_outline_width (glestext);
  update_text (glestext);
}
