/* places-glib: A GObject wrapper for the Mozilla Places API
 *
 * Copyright (C) 2009  Intel Corporation
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <glib-object.h>
#include <dbus/dbus-glib.h>
#include <new>
#include <nsINavHistoryService.h>
#include <nsIFaviconService.h>
#include <nsIAutoCompleteSearch.h>
#include <nsIAutoCompleteResult.h>
#include <nsIIOService.h>
#include <nsIGlobalHistory2.h>
#include <nsToolkitCompsCID.h>
#include <nsNetCID.h>
#include <nsServiceManagerUtils.h>
#include <nsCOMPtr.h>
#include <nsStringAPI.h>
#include <nsMemory.h>

#include "places-history.h"
#include "places-error-private.h"
#include "places-service.h"
#include "places-marshal.h"

/* D-Bus method implementations */

static gboolean places_history_add_uri (PlacesHistory *self,
                                        const gchar *uri,
                                        GError **error);

static gboolean places_history_set_page_title (PlacesHistory *self,
                                               const gchar *uri,
                                               const gchar *title,
                                               GError **error);

static gboolean places_history_start_ac_search (PlacesHistory *self,
                                                const gchar *search_str,
                                                guint32 *search_id,
                                                GError **error);

static gboolean places_history_stop_ac_search (PlacesHistory *self,
                                               guint32 search_id,
                                               GError **error);

static gboolean places_history_set_favicon_url (PlacesHistory *self,
                                                const gchar *page_uri_str,
                                                const gchar *favicon_uri_str,
                                                GError **error);

static gboolean places_history_set_default_favicon_url
                                               (PlacesHistory *self,
                                                const gchar *page_uri_str,
                                                GError **error);

static gboolean places_history_get_favicon (PlacesHistory *self,
                                            const gchar *page_uri,
                                            gboolean download,
                                            DBusGMethodInvocation *context);

#include "places-history-glue.h"

/* End D-Bus method implementations */

static void places_history_finalize (GObject *object);

G_DEFINE_TYPE (PlacesHistory, places_history, G_TYPE_OBJECT);

#define PLACES_HISTORY_GET_PRIVATE(obj) \
  (G_TYPE_INSTANCE_GET_PRIVATE ((obj), PLACES_TYPE_HISTORY, \
                                PlacesHistoryPrivate))

class PlacesHistoryObserver : public nsIAutoCompleteObserver,
                              public nsINavHistoryObserver
{
public:
  NS_DECL_ISUPPORTS
  NS_DECL_NSIAUTOCOMPLETEOBSERVER
  NS_DECL_NSINAVHISTORYOBSERVER

  PlacesHistory *history;
};

struct _PlacesHistoryPrivate
{
  guint32 ac_search_id;
  gboolean in_ac_search;

  /* Number of history items that we have reported back from the
     result set so far */
  guint n_results_reported;

  gboolean history_observer_added;
  PlacesHistoryObserver observer;

  GSList *favicon_closures;
};

struct PlacesHistoryFaviconClosure
{
  PlacesHistoryFaviconClosure (nsIURI *page_uri, nsIURI *favicon_uri,
                               DBusGMethodInvocation *context)
    : page_uri (page_uri),
      favicon_uri (favicon_uri),
      context (context)
  {
  }

  nsCOMPtr<nsIURI> page_uri, favicon_uri;
  DBusGMethodInvocation *context;
};

enum
  {
    AC_RESULT_RECEIVED_SIGNAL,

    LAST_SIGNAL
  };

static guint history_signals[LAST_SIGNAL] = { 0, };

NS_IMPL_QUERY_INTERFACE2 (PlacesHistoryObserver,
                          nsIAutoCompleteObserver,
                          nsINavHistoryObserver)

/* We don't want to implement reference counting for
   PlacesHistoryObserver because it is entirely owned by the
   PlacesHistory instance */
NS_IMETHODIMP_(nsrefcnt)
PlacesHistoryObserver::AddRef ()
{
  return 1;
}

NS_IMETHODIMP_(nsrefcnt)
PlacesHistoryObserver::Release ()
{
  return 1;
}

static void
places_history_favicon_closure_destroy (PlacesHistoryFaviconClosure *data)
{
  if (data->context)
    {
      GError *error = NULL;
      places_error_set_from_nsresult (NS_ERROR_NOT_AVAILABLE, &error);
      dbus_g_method_return_error (data->context, error);
      g_error_free (error);
    }

  data->~PlacesHistoryFaviconClosure ();
  g_slice_free (PlacesHistoryFaviconClosure, data);
}

NS_IMETHODIMP
PlacesHistoryObserver::OnSearchResult (nsIAutoCompleteSearch *search,
                                       nsIAutoCompleteResult *result)
{
  nsresult rv;
  PlacesHistoryPrivate *priv = history->priv;
  PRUint16 search_result;
  PRUint32 match_count;

  rv = result->GetSearchResult (&search_result);

  if (NS_FAILED (rv))
    {
      g_warning ("GetSearchResult failed: 0x%x\n", rv);
      return rv;
    }

  rv = result->GetMatchCount (&match_count);

  if (NS_FAILED (rv))
    {
      g_warning ("GetMatchCount failed: 0x%x\n", rv);
      return rv;
    }

  if (search_result == nsIAutoCompleteResult::RESULT_SUCCESS
      || search_result == nsIAutoCompleteResult::RESULT_SUCCESS_ONGOING)
    while (priv->n_results_reported < match_count)
      {
        nsAutoString value, comment, image;

        if (NS_SUCCEEDED (result->GetValueAt (priv->n_results_reported, value))
            && NS_SUCCEEDED (result->GetCommentAt (priv->n_results_reported,
                                                   comment)))
          {
            NS_ConvertUTF16toUTF8 value_utf8 (value);
            NS_ConvertUTF16toUTF8 comment_utf8 (comment);

            g_signal_emit (history,
                           history_signals[AC_RESULT_RECEIVED_SIGNAL], 0,
                           priv->ac_search_id,
                           value_utf8.get (),
                           comment_utf8.get ());
          }

        priv->n_results_reported++;
      }

  return NS_OK;
}

NS_IMETHODIMP
PlacesHistoryObserver::OnBeginUpdateBatch ()
{
  return NS_ERROR_NOT_IMPLEMENTED;
}

NS_IMETHODIMP
PlacesHistoryObserver::OnEndUpdateBatch ()
{
  return NS_ERROR_NOT_IMPLEMENTED;
}

NS_IMETHODIMP
PlacesHistoryObserver::OnVisit (nsIURI *uri,
                                PRInt64 visit_id,
                                PRTime when,
                                PRInt64 session_id,
                                PRInt64 referring_id,
                                PRUint32 transition_type,
                                PRUint32 *added NS_OUTPARAM)
{
  return NS_ERROR_NOT_IMPLEMENTED;
}

NS_IMETHODIMP
PlacesHistoryObserver::OnTitleChanged (nsIURI *uri,
                                       const nsAString &page_title)
{
  return NS_ERROR_NOT_IMPLEMENTED;
}

NS_IMETHODIMP
PlacesHistoryObserver::OnDeleteURI (nsIURI *uri)
{
  return NS_ERROR_NOT_IMPLEMENTED;
}

NS_IMETHODIMP
PlacesHistoryObserver::OnClearHistory ()
{
  return NS_ERROR_NOT_IMPLEMENTED;
}

NS_IMETHODIMP
PlacesHistoryObserver::OnPageChanged (nsIURI *uri,
                                      PRUint32 what_changed,
                                      const nsAString &value)
{
  PlacesHistoryPrivate *priv = history->priv;
  nsresult rv;

  if (what_changed == ATTRIBUTE_FAVICON)
    {
      GSList *next;

      nsCOMPtr<nsIFaviconService> favicon_service
        = do_GetService (NS_FAVICONSERVICE_CONTRACTID, &rv);

      if (NS_FAILED (rv))
        return rv;

      /* Look for a matching URI in the list of closures we are
         waiting for */
      for (GSList *l = priv->favicon_closures; l; l = next)
        {
          PlacesHistoryFaviconClosure *data
            = (PlacesHistoryFaviconClosure *) l->data;
          PRBool uri_equals;

          next = l->next;

          rv = data->page_uri->Equals (uri, &uri_equals);

          if (NS_SUCCEEDED (rv) && uri_equals)
            {
              nsCAutoString mime_type;
              PRUint32 data_len;
              PRUint8 *icon_data;

              rv = favicon_service->GetFaviconData (data->favicon_uri,
                                                    mime_type,
                                                    &data_len, &icon_data);

              if (NS_FAILED (rv))
                {
                  GError *error = NULL;
                  places_error_set_from_nsresult (rv, &error);
                  dbus_g_method_return_error (data->context, error);
                  g_error_free (error);
                }
              else
                {
                  /* Copy the data into a GArray to pass to D-BUS */
                  GArray *data_array =
                    g_array_sized_new (FALSE, FALSE, sizeof (guint8), data_len);

                  g_array_append_vals (data_array, icon_data, data_len);

                  dbus_g_method_return (data->context,
                                        mime_type.get (),
                                        data_array);

                  g_array_free (data_array, TRUE);
                  nsMemory::Free (icon_data);
                }

              data->context = NULL;

              priv->favicon_closures
                = g_slist_remove (priv->favicon_closures, data);
              places_history_favicon_closure_destroy (data);
            }
        }
    }

  return NS_OK;
}

NS_IMETHODIMP
PlacesHistoryObserver::OnPageExpired (nsIURI *uri,
                                      PRTime visit_time,
                                      PRBool whole_entry)
{
  return NS_ERROR_NOT_IMPLEMENTED;
}

static void
places_history_class_init (PlacesHistoryClass *klass)
{
  GObjectClass *gobject_class = (GObjectClass *) klass;

  gobject_class->finalize = places_history_finalize;

  history_signals[AC_RESULT_RECEIVED_SIGNAL] =
    g_signal_new ("ac-result-received",
		  G_TYPE_FROM_CLASS (gobject_class),
		  G_SIGNAL_RUN_LAST,
		  G_STRUCT_OFFSET (PlacesHistoryClass, ac_result_received),
		  NULL, NULL,
                  places_marshal_VOID__UINT_STRING_STRING,
                  G_TYPE_NONE, 3,
                  G_TYPE_UINT,
                  G_TYPE_STRING,
                  G_TYPE_STRING);

  g_type_class_add_private (klass, sizeof (PlacesHistoryPrivate));

  dbus_g_object_type_install_info (PLACES_TYPE_HISTORY,
                                   &dbus_glib_places_history_object_info);
}

static void
places_history_init (PlacesHistory *self)
{
  PlacesHistoryPrivate *priv;
  DBusGConnection *connection;
  GError *error = NULL;

  priv = self->priv = PLACES_HISTORY_GET_PRIVATE (self);

  priv->ac_search_id = 1;

  /* 'placement new' to call the constructor for
     PlacesHistoryPrivate */
  new (reinterpret_cast<void *> (priv)) PlacesHistoryPrivate;

  if ((connection = dbus_g_bus_get (DBUS_BUS_SESSION, &error)) == NULL)
    {
      g_warning ("Error connecting to session bus: %s", error->message);
      g_error_free (error);
    }
  else
    {
      dbus_g_connection_register_g_object (connection,
                                           PLACES_SERVICE_HISTORY_PATH,
                                           G_OBJECT (self));
      dbus_g_connection_unref (connection);
    }

  priv->observer.history = self;
}

static void
places_history_finalize (GObject *object)
{
  PlacesHistory *self = (PlacesHistory *) object;
  PlacesHistoryPrivate *priv = self->priv;
  nsresult rv;

  places_history_stop_ac_search (self, priv->ac_search_id, NULL);

  if (priv->history_observer_added)
    {
      /* Unregister the observer for history notifications as the observer
         object is going to be destroyed */
      nsCOMPtr<nsINavHistoryService> history_service
        = do_GetService (NS_NAVHISTORYSERVICE_CONTRACTID, &rv);
      if (NS_SUCCEEDED (rv))
        history_service->RemoveObserver (&priv->observer);
    }

  g_slist_foreach (priv->favicon_closures,
                   (GFunc) places_history_favicon_closure_destroy, NULL);
  g_slist_free (priv->favicon_closures);
  priv->favicon_closures = NULL;

  /* Explicitly call the destructor for the private data (so that it
     destructs without trying to free the memory) */
  priv->~PlacesHistoryPrivate ();

  G_OBJECT_CLASS (places_history_parent_class)->finalize (object);
}

PlacesHistory *
places_history_new (void)
{
  PlacesHistory *self = (PlacesHistory *) g_object_new (PLACES_TYPE_HISTORY,
                                                        NULL);

  return self;
}

static nsCOMPtr<nsIURI>
places_history_string_to_uri (const gchar *uri_string, GError **error)
{
  nsresult rv;
  nsCOMPtr<nsIIOService> io_service
    = do_GetService (NS_IOSERVICE_CONTRACTID, &rv);

  if (NS_FAILED (rv))
    {
      places_error_set_from_nsresult (rv, error);
      return NULL;
    }

  nsDependentCString uri_cstring (uri_string);

  nsCOMPtr<nsIURI> uri;
  rv = io_service->NewURI (uri_cstring, NULL, NULL, getter_AddRefs (uri));

  if (NS_FAILED (rv))
    {
      places_error_set_from_nsresult (rv, error);
      return NULL;
    }

  return uri;
}

static gboolean
places_history_add_uri (PlacesHistory *self,
                        const gchar *uri,
                        GError **error)
{
  nsresult rv;

  g_return_val_if_fail (PLACES_IS_HISTORY (self), FALSE);

  nsCOMPtr<nsIGlobalHistory2> global_history2
    = do_GetService (NS_NAVHISTORYSERVICE_CONTRACTID, &rv);

  if (NS_FAILED (rv))
    {
      places_error_set_from_nsresult (rv, error);
      return FALSE;
    }

  nsCOMPtr<nsIURI> nsuri = places_history_string_to_uri (uri, error);
  if (!nsuri)
    return FALSE;

  nsCOMPtr<nsIURI> referrer_nsuri;

  /* The redirect parameter is deprecated so we always set it to false */
  rv = global_history2->AddURI (nsuri, PR_FALSE, PR_FALSE,
                                referrer_nsuri);

  if (NS_FAILED (rv))
    {
      places_error_set_from_nsresult (rv, error);
      return FALSE;
    }

  return TRUE;
}

static gboolean
places_history_set_page_title (PlacesHistory *self,
                               const gchar *uri,
                               const gchar *title,
                               GError **error)
{
  nsresult rv;

  nsCOMPtr<nsIGlobalHistory2> global_history2
    = do_GetService (NS_NAVHISTORYSERVICE_CONTRACTID, &rv);

  if (NS_FAILED (rv))
    {
      places_error_set_from_nsresult (rv, error);
      return FALSE;
    }

  nsCOMPtr<nsIURI> nsuri = places_history_string_to_uri (uri, error);
  if (!nsuri)
    return FALSE;

  nsDependentCString title_cstring (title);
  NS_ConvertUTF8toUTF16 title_string (title_cstring);

  /* The redirect parameter is deprecated so we always set it to false */
  rv = global_history2->SetPageTitle (nsuri, title_string);

  if (NS_FAILED (rv))
    {
      places_error_set_from_nsresult (rv, error);
      return FALSE;
    }

  return TRUE;
}

static gboolean
places_history_start_ac_search (PlacesHistory *self,
                                const gchar *search_str,
                                guint32 *search_id,
                                GError **error)
{
  PlacesHistoryPrivate *priv = self->priv;
  nsresult rv;

  nsCOMPtr<nsIAutoCompleteSearch> auto_complete
    = do_GetService (NS_NAVHISTORYSERVICE_CONTRACTID, &rv);

  if (NS_FAILED (rv))
    {
      places_error_set_from_nsresult (rv, error);
      return FALSE;
    }

  /* Stop any existing searches */
  rv = auto_complete->StopSearch ();
  if (NS_FAILED (rv))
    {
      places_error_set_from_nsresult (rv, error);
      return FALSE;
    }

  priv->n_results_reported = 0;

  rv = auto_complete->StartSearch (NS_ConvertUTF8toUTF16 (search_str),
                                   EmptyString (),
                                   NULL,
                                   &priv->observer);
  if (NS_FAILED (rv))
    {
      places_error_set_from_nsresult (rv, error);
      return FALSE;
    }

  priv->in_ac_search = TRUE;

  *search_id = priv->ac_search_id;

  return TRUE;
}

static gboolean
places_history_stop_ac_search (PlacesHistory *self,
                               guint32 search_id,
                               GError **error)
{
  nsresult rv;

  g_return_val_if_fail (PLACES_IS_HISTORY (self), FALSE);

  PlacesHistoryPrivate *priv = self->priv;

  if (priv->in_ac_search && priv->ac_search_id == search_id)
    {
      nsCOMPtr<nsIAutoCompleteSearch> auto_complete
        = do_GetService (NS_NAVHISTORYSERVICE_CONTRACTID, &rv);

      if (NS_FAILED (rv))
        {
          places_error_set_from_nsresult (rv, error);
          return FALSE;
        }

      rv = auto_complete->StopSearch ();
      if (NS_FAILED (rv))
        {
          places_error_set_from_nsresult (rv, error);
          return FALSE;
        }

      priv->in_ac_search = FALSE;

      /* Increase the search id so that we know to ignore any results
         for previous queries. In the unlikely event of wrap-around,
         we don't want to reuse search id 0 */
      if (++priv->ac_search_id == 0)
        priv->ac_search_id = 1;
    }

  return TRUE;
}

static gboolean
places_history_set_favicon_url (PlacesHistory *self,
                                const gchar *page_uri_str,
                                const gchar *favicon_uri_str,
                                GError **error)
{
  nsresult rv;

  g_return_val_if_fail (PLACES_IS_HISTORY (self), FALSE);

  nsCOMPtr<nsIURI> favicon_uri, page_uri;

  nsCOMPtr<nsIFaviconService> favicon_service
    = do_GetService (NS_FAVICONSERVICE_CONTRACTID, &rv);

  if (NS_FAILED (rv))
    {
      places_error_set_from_nsresult (rv, error);
      return FALSE;
    }

  page_uri = places_history_string_to_uri (page_uri_str, error);
  if (!page_uri)
    return FALSE;
  favicon_uri = places_history_string_to_uri (favicon_uri_str, error);
  if (!favicon_uri)
    return FALSE;

  rv = favicon_service->SetFaviconUrlForPage (page_uri, favicon_uri);

  if (NS_FAILED (rv))
    {
      places_error_set_from_nsresult (rv, error);
      return FALSE;
    }

  return TRUE;
}

static gboolean
places_history_set_default_favicon_url (PlacesHistory *self,
                                        const gchar *page_uri_str,
                                        GError **error)
{
  nsresult rv;

  g_return_val_if_fail (PLACES_IS_HISTORY (self), FALSE);

  nsCOMPtr<nsIURI> favicon_uri, page_uri;
  nsCAutoString uri_path;

  nsCOMPtr<nsIFaviconService> favicon_service
    = do_GetService (NS_FAVICONSERVICE_CONTRACTID, &rv);

  if (NS_FAILED (rv))
    {
      places_error_set_from_nsresult (rv, error);
      return FALSE;
    }

  page_uri = places_history_string_to_uri (page_uri_str, error);
  if (!page_uri)
    return FALSE;

  /* Make a new URL with the pre_path of the page URI + "/favicon.uri" */
  rv = page_uri->GetPrePath (uri_path);
  if (NS_FAILED (rv))
    {
      places_error_set_from_nsresult (rv, error);
      return FALSE;
    }

  uri_path.AppendLiteral ("/favicon.ico");

  favicon_uri = places_history_string_to_uri (uri_path.get (), error);
  if (!favicon_uri)
    return FALSE;

  rv = favicon_service->SetFaviconUrlForPage (page_uri, favicon_uri);

  if (NS_FAILED (rv))
    {
      places_error_set_from_nsresult (rv, error);
      return FALSE;
    }

  return TRUE;
}

static gboolean
places_history_get_favicon (PlacesHistory *self,
                            const gchar *page_uri_str,
                            gboolean download,
                            DBusGMethodInvocation *context)
{
  nsresult rv;
  GError *error = NULL;

  g_return_val_if_fail (PLACES_IS_HISTORY (self), FALSE);

  PlacesHistoryPrivate *priv = self->priv;
  nsCOMPtr<nsIURI> favicon_uri, page_uri;

  nsCOMPtr<nsIFaviconService> favicon_service
    = do_GetService (NS_FAVICONSERVICE_CONTRACTID, &rv);

  if (NS_FAILED (rv))
    goto rv_error;

  page_uri = places_history_string_to_uri (page_uri_str, &error);
  if (!page_uri)
    goto error;

  /* Get the URL for the favicon of this page or bail out if it hasn't
     been set yet */
  rv = favicon_service->GetFaviconForPage (page_uri,
                                           getter_AddRefs (favicon_uri));
  if (NS_FAILED (rv))
    goto rv_error;

  /* Check if places already has the data for the icon cached */
  {
    nsCAutoString mime_type;
    PRUint32 data_len;
    PRUint8 *data = nsnull;

    rv = favicon_service->GetFaviconData (favicon_uri, mime_type,
                                          &data_len, &data);

    /* The documentation says that GetFaviconData will
       return NS_ERROR_NOT_AVAILABLE is there is no data, but it
       actually appears to return an empty array instead */
    if (NS_SUCCEEDED (rv) && data_len == 0)
      {
        if (data != nsnull)
          nsMemory::Free (data);
        rv = NS_ERROR_NOT_AVAILABLE;
      }

    /* If it isn't available then start downloading it
       asynchronously */
    if (rv == NS_ERROR_NOT_AVAILABLE && download)
      {
        /* Make sure we are observing page changes to catch when the
           favicon has loaded */
        if (!priv->history_observer_added)
          {
            /* Register the observer for history notifications */
            nsCOMPtr<nsINavHistoryService> history_service
              = do_GetService (NS_NAVHISTORYSERVICE_CONTRACTID, &rv);
            if (NS_FAILED (rv))
              goto rv_error;
            history_service->AddObserver (&priv->observer, PR_FALSE);

            priv->history_observer_added = TRUE;
          }

        PlacesHistoryFaviconClosure *data
          = g_slice_new (PlacesHistoryFaviconClosure);
        new (reinterpret_cast<void *> (data))
          PlacesHistoryFaviconClosure (page_uri, favicon_uri, context);
        priv->favicon_closures = g_slist_prepend (priv->favicon_closures, data);

        rv = favicon_service->SetAndLoadFaviconForPage (page_uri, favicon_uri,
                                                        PR_FALSE);
        if (NS_FAILED (rv))
          {
            priv->favicon_closures
              = g_slist_remove (priv->favicon_closures, data);
            data->context = NULL;
            places_history_favicon_closure_destroy (data);
            goto rv_error;
          }
      }
    else if (NS_FAILED (rv))
      goto rv_error;
    else
      {
        /* Copy the data into a GArray to pass to D-BUS */
        GArray *data_array =
          g_array_sized_new (FALSE, FALSE, sizeof (guint8), data_len);

        g_array_append_vals (data_array, data, data_len);

        dbus_g_method_return (context,
                              mime_type.get (),
                              data_array);

        g_array_free (data_array, TRUE);
        nsMemory::Free (data);
      }
  }

  return TRUE;

 rv_error:
  places_error_set_from_nsresult (rv, &error);
 error:
  dbus_g_method_return_error (context, error);
  g_error_free (error);
  return TRUE;
}
