#
# This file is part of Python Terra
# Copyright (C) 2007-2009 Instituto Nokia de Tecnologia
# Contact: Renato Chencarek <renato.chencarek@openbossa.org>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#

import etk
import edje.decorators
import evas.decorators

import logging
import base

__all__ = ("Modal", "ModalSlider", "SliderContents", "SliderContentList",
           "ModalMessage", "ModalMessageConfim" )

log = logging.getLogger("terra.ui.modal")


class Modal(base.EdjeSettingsWidget):
    def __init__(self, parent, title, theme=None, hborder=0, vborder=0):
        base.EdjeSettingsWidget.__init__(self, parent, title, "modal", theme)

        self.callback_animate_finished = None

        if hborder < 0 or vborder < 0:
            raise ValueError("Border values must not be negative")

        self.hborder = hborder
        self.vborder = vborder
        self._on_parent_resize(parent)

        parent.on_resize_add(self._on_parent_resize)

    def _on_parent_resize(self, parent):
        x, y, w, h = parent.geometry

        if self.hborder:
            x += self.hborder
            w -= 2*self.hborder

        if self.vborder:
            y += self.vborder
            h -= 2*self.vborder

        self.geometry = (x, y, w, h)

    def hide(self, end_callback=None):
        if end_callback is not None:
            def cb(*ignored):
                self.signal_callback_del("modal,animate,hide,finished", "", cb)
                end_callback()
            self.signal_callback_add("modal,animate,hide,finished", "", cb)

        self.signal_emit("modal,hide", "")

    def half_expand(self):
        self.signal_emit("modal,expand,half", "")

    @edje.decorators.signal_callback("modal,animate,show,finished", "")
    def _cb_on_animate_show_finished(self, *ignored):
        if self.callback_animate_finished:
            self.callback_animate_finished()

    @edje.decorators.signal_callback("modal,animate,hide,finished", "")
    def _cb_on_animate_hide_finished(self, *ignored):
        base.EdjeSettingsWidget.hide(self)



class ModalSlider(Modal):
    HEADER_AREA_HEIGHT = 70
    BUTTON_BOX_HEIGHT  = 70

    def __init__(self, parent, title, theme=None, hborder=0, vborder=0):
        Modal.__init__(self, parent, title, theme, hborder, vborder)

        self.slider_contents = SliderContents(self.evas, self, theme)
        self._init_header()
        self._init_button_box()

        self.slider_contents.show()
        self.contents_set(self.slider_contents)

    def _init_header(self):
        self.header_vbox = etk.VBox()
        self.title_separator = etk.HSeparator()
        self.header_vbox.append(self.title_separator, etk.VBox.START,
                                etk.VBox.FILL, 0)
        self.header_vbox.show()

        self.header_embed = etk.Embed(self.evas)
        self.slider_contents.part_swallow("header_area",
                                          self.header_embed.object)

    def _init_button_box(self):
        self.bottom_vbox = etk.VBox()
        self.separator = etk.HSeparator()
        self.bottom_vbox.append(self.separator, etk.VBox.START,
                                etk.VBox.FILL, 0)

        self.bbox = etk.HBox()
        self.bbox.border_width_set(15)
        self.bbox.show()
        self.bottom_vbox.append(self.bbox, etk.VBox.START,
                                etk.VBox.EXPAND_FILL, 0)
        self.bottom_vbox.show()

        self.bbox_embed = etk.Embed(self.evas)
        self.slider_contents.part_swallow("button_box", self.bbox_embed.object)

    @evas.decorators.del_callback
    def _destroy_embeds(self):
        self.header_vbox.destroy()
        self.header_vbox = None
        self.header_embed.destroy()
        self.header_embed = None
        self.bbox_embed.destroy()
        self.bbox_embed = None
        self.slider_contents.delete()
        self.slider_contents = None

    def header_area_hide(self):
        w, h = self.slider_contents.size
        self.header_embed.hide()
        header_obj = self.slider_contents.part_swallow_get("header_area")
        edje.extern_object_max_size_set(header_obj, w, 0)

    def header_area_show(self):
        w, h = self.slider_contents.size
        header_obj = self.slider_contents.part_swallow_get("header_area")
        edje.extern_object_min_size_set(header_obj, w, self.HEADER_AREA_HEIGHT)

        if not self.header_embed.children_get():
            self.header_embed.add(self.header_vbox)

        self.header_embed.show()

    def button_box_hide(self):
        w, h = self.slider_contents.size
        self.bbox_embed.hide()
        bbox_obj = self.slider_contents.part_swallow_get("button_box")
        edje.extern_object_max_size_set(bbox_obj, w, 0)

    def button_box_show(self):
        w, h = self.slider_contents.size
        bbox_obj = self.slider_contents.part_swallow_get("button_box")
        edje.extern_object_min_size_set(bbox_obj, w, self.BUTTON_BOX_HEIGHT)

        if not self.bbox_embed.children_get():
            self.bbox_embed.add(self.bottom_vbox)

        self.bbox_embed.show()

    def button_add(self, label, show=True):
        button = etk.Button(label=label)
        button.focusable = False
        button.on_clicked(self._on_button_clicked)
        if show:
            button.show()

        if not self.bbox.children:
            self.bbox.append(button, etk.HBox.END, etk.HBox.FILL, 0)
        else:
            self.bbox.append(button, etk.HBox.START, etk.HBox.FILL, 0)

        return button

    def _on_button_clicked(self, button):
        log.debug("Button %s clicked", button.label_get())

    def __getattr__(self, name):
        try:
            return getattr(self.slider_contents, name)
        except AttributeError:
            raise AttributeError("%s object has no attribute '%s'" % (type(self), name))

    @evas.decorators.focus_in_callback
    def _focus_in(self):
        self.slider_contents.focus = True


class SliderContents(base.EdjeWidget):
    def __init__(self, canvas, parent=None, theme=None):
        base.EdjeWidget.__init__(self, canvas, "slider_contents", parent, theme)

        self.animating = False

    def _emit_signal(self, signal_name, animate=True):
        if self.animating and animate:
            return

        self.animating = animate
        self.signal_emit(signal_name, "")

    def reset(self):
        self._emit_signal("reset", animate=False)

    def move_right(self, end_callback=None):
        if end_callback is not None:
            def cb(*ignored):
                self.signal_callback_del("move,right,finished", "", cb)
                end_callback()

            self.signal_callback_add("move,right,finished", "", cb)
        self._emit_signal("move,right")

    def move_left(self, end_callback=None):
        if end_callback is not None:
            def cb(*ignored):
                self.signal_callback_del("move,left,finished", "", cb)
                end_callback()

            self.signal_callback_add("move,left,finished", "", cb)
        self._emit_signal("move,left")

    def _get_all_contents(self):
        left   = self.left_contents_get()
        center = self.center_contents_get()
        right  = self.right_contents_get()

        return (left, center, right)

    def unset_all_contents(self):
        self.left_contents_set(None)
        self.center_contents_set(None)
        self.right_contents_set(None)

    def _contents_set(self, part_name, contents):
        content = self.part_swallow_get(part_name)

        if content:
            self.part_unswallow(content)
            content.hide()

        if contents:
            self.part_swallow(part_name, contents)

    def left_contents_get(self):
        return self.part_swallow_get("contents_left")

    def left_contents_set(self, contents):
        self._contents_set("contents_left", contents)

    left_contents = property(left_contents_get, left_contents_set)

    def center_contents_get(self):
        return self.part_swallow_get("contents_center")

    def center_contents_set(self, contents):
        self._contents_set("contents_center", contents)
        if contents:
            contents.focus = True

    center_contents = property(center_contents_get, center_contents_set)

    def right_contents_get(self):
        return self.part_swallow_get("contents_right")

    def right_contents_set(self, contents):
        self._contents_set("contents_right", contents)

    right_contents = property(right_contents_get, right_contents_set)

    @edje.decorators.signal_callback("move,left,finished", "")
    def _cb_on_move_left_finished(self, *ignored):
        left, center, right = self._get_all_contents()
        self.unset_all_contents()

        self.left_contents_set(center)
        self.center_contents_set(right)
        self.reset()

    @edje.decorators.signal_callback("move,right,finished", "")
    def _cb_on_move_right_finished(self, *ignored):
        left, center, right = self._get_all_contents()
        self.unset_all_contents()

        self.right_contents_set(center)
        self.center_contents_set(left)
        self.reset()

    @evas.decorators.del_callback
    def _delete_callback(self):
        left, center, right = self._get_all_contents()
        self.unset_all_contents()

        if left:
            left.delete()

        if center:
            center.delete()

        if right:
            right.delete()

    @evas.decorators.focus_in_callback
    def _focus_in(self):
        self.center_contents.focus = True
        self.do_on_focus()

    def do_on_focus(self):
        pass


class SliderContentList(etk.Embed):
    def __init__(self, evas_obj):
        etk.Embed.__init__(self, evas_obj)

        self.list = self._create_list()
        self.list.show()
        self.add(self.list)

        self.on_key_down(self._key_down_cb)
        self.on_key_up(self._key_up_cb)

        self.callback_escape = None

    def _focus_set(self, value):
        self.object.focus = value

    focus = property(fset=_focus_set)

    def _key_down_cb(self, o, ev):
        return self.handle_key_down(ev)

    def _key_up_cb(self, o, ev):
        return self.handle_key_up(ev)

    def handle_key_down(self, ev):
        if ev.key == "Escape":
            if self.callback_escape:
                self.callback_escape()
            return False
        return True

    def handle_key_up(self, ev):
        return True

    def _create_list(self):
        columns = self.columns_get()
        model = self.model_get()

        if not model:
            model = etk.ListModel()

        list = etk.List(model=model, selectable=etk.List.NOT_SELECTABLE,
                        columns=columns)
        list.focusable = False
        return list

    def model_get(self):
        return None

    def columns_get(self):
        raise NotImplementedError("%s.columns_get() not implemented." % \
                                      self.__class__.__name__)

    def item_append(self, *args):
        self.list.model.append(*args)


# XXX: We should extract stuff from this like we did in ModalRich, and then
# derive just adding the visible UI.
class ModalMessage(Modal):
    def __init__(self, parent, message, theme=None, hborder=0, vborder=0):
        Modal.__init__(self, parent, "", theme, hborder, vborder)

        self._init_embed()
        self.callback_clicked = None
        self.callback_escape = None
        self.message = message

    def _init_embed(self):
        self.panel = base.EdjeWidget(self.evas, "modal/content/message", self)
        self.panel.part_text_set("button:text", "   OK   ")
        self.panel.signal_callback_add("button,clicked", "button:", self.clicked)
        self.contents_set(self.panel)
        self.panel.on_key_up_add(self._key_up_cb)
        self.panel.on_key_down_add(self._key_down_cb)

    def _message_set(self, value):
        self.panel.part_text_set("modal_text", value)

    def _message_get(self):
        return self.panel.part_text_get("modal_text")

    message = property(_message_get, _message_set)

    def clicked(self, *ignored):
        if self.callback_clicked:
            self.callback_clicked()

    def _key_down_cb(self, o, ev):
        return self.handle_key_down(ev)

    def _key_up_cb(self, o, ev):
        return self.handle_key_up(ev)

    def handle_key_down(self, ev):
        if ev.key == "Escape":
            if self.callback_escape:
                self.callback_escape()
            return False
        return True

    def handle_key_up(self, ev):
        return True

    @evas.decorators.focus_in_callback
    def _focus_in(self):
        self.panel.focus = True
        self.do_on_focus()

    def do_on_focus(self):
        pass

    @evas.decorators.del_callback
    def _destroy_contents(self):
        self.panel.delete()


class ModalMessageConfirm(Modal):
    def __init__(self, parent, message, theme=None, hborder=0, vborder=0):
        Modal.__init__(self, parent, "", theme, hborder, vborder)

        self._init_embed()
        self.callback_clicked = None
        self.callback_escape = None
        self.message = message

    def _init_embed(self):
        self.panel = base.EdjeWidget(self.evas, "modal/content/confirm", self)
        self.panel.part_text_set("button_yes:text", "   OK   ")
        self.panel.signal_callback_add("button,clicked", "button_yes:", self.clicked_yes)
        self.panel.part_text_set("button_no:text", " Cancel ")
        self.panel.signal_callback_add("button,clicked", "button_no:", self.clicked_no)
        self.contents_set(self.panel)
        self.panel.on_key_up_add(self._key_up_cb)
        self.panel.on_key_down_add(self._key_down_cb)

    def _message_set(self, value):
        self.panel.part_text_set("modal_text", value)

    def _message_get(self):
        return self.panel.part_text_get("modal_text")

    message = property(_message_get, _message_set)

    def clicked_yes(self, *ignored):
        if self.callback_clicked_yes:
            self.callback_clicked_yes()

    def clicked_no(self, *ignored):
        if self.callback_clicked_no:
            self.callback_clicked_no()

    def _key_down_cb(self, o, ev):
        return self.handle_key_down(ev)

    def _key_up_cb(self, o, ev):
        return self.handle_key_up(ev)

    def handle_key_down(self, ev):
        if ev.key == "Escape":
            if self.callback_escape:
                self.callback_escape()
            return False
        return True

    def handle_key_up(self, ev):
        return True

    @evas.decorators.focus_in_callback
    def _focus_in(self):
        self.panel.focus = True
        self.do_on_focus()

    def do_on_focus(self):
        pass

    @evas.decorators.del_callback
    def _destroy_contents(self):
        self.panel.delete()
