#
# This file is part of Python Terra
# Copyright (C) 2007-2009 Instituto Nokia de Tecnologia
# Contact: Renato Chencarek <renato.chencarek@openbossa.org>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#

import logging

import base
import etk
import edje.decorators
import evas.decorators

__all__ = ("Panel", "PanelContent", "PanelContentList", "PanelRich",
           "PanelRichList")

log = logging.getLogger("terra.ui.panel")

def data_as_int_get(obj, key, default):
    try:
        return int(obj.data_get(key))
    except TypeError, e:
        return default


class Panel(base.EdjeSettingsWidget):
    """Stacked panels over the MainWindow.

    This is the base class for panels that are stacked over MainWindow, which
    are used for global Settings and per-screen Options.
    """
    def __init__(self, main_window, title, theme=None):
        base.EdjeSettingsWidget.__init__(self, main_window, title,
                                         "panel", theme)
        self.panel_type = self.SETTINGS

    @edje.decorators.signal_callback("action,clicked", "panel,back")
    def on_blocker_clicked(self, *ignored):
        self._parent_widget.callback_panel_back()

    SETTINGS, OPTIONS = range(2)

    def _panel_type_set(self, value):
        w, h = self._parent_widget.size

        if value == self.SETTINGS:
            self._panel_type = value
            self.width = data_as_int_get(self, "settings_width", w / 2)
            self.resize(self.width, h)

        else: # For now, fallback to options
            self._panel_type = value
            self.width = data_as_int_get(self, "options_width", w / 3)
            self.resize(self.width, h)

    def _panel_type_get(self):
        return self._panel_type

    panel_type = property(_panel_type_get, _panel_type_set)

    @evas.decorators.key_up_callback
    def _cb_on_key_up(self, event):
        self.handle_key_up(event)

    def handle_key_up(self, event):
        pass

    @evas.decorators.key_down_callback
    def _cb_on_key_down(self, event):
        self.handle_key_down(event)

    def handle_key_down(self, event):
        pass


class PanelContent(etk.Embed):
    """Base class for panel contents.

    Inherits from etk.Embed which is an EvasObject and also an Etk
    Container, since most UI for settings and options use Etk library
    for components.

    The basic layout for every panel, after the title a L{header}
    (usually for text or an textentry), optionally a L{child} which is
    the center of the panel (usually a list or an textentry) and a
    L{button_box}.

    Both header area and button box are of type etk.HBox. Use
    L{header_area_append()} and L{button_box_add()} to add widgets in
    these boxes.
    """
    def __init__(self, evas_obj):
        etk.Embed.__init__(self, evas_obj)

        self.handle_key_down = None
        self.handle_key_up = None

        self._vbox = etk.VBox()

        self.header_area = etk.HBox()
        self.header_area.border_width_set(15)
        self._vbox.append(self.header_area, etk.VBox.START, etk.VBox.FILL, 0)

        self._child = self._child_get()
        if self._child:
            self._child.show()
            self._vbox.append(self._child, etk.VBox.START,
                              etk.VBox.EXPAND_FILL, 0)

        self.button_box = etk.HBox()
        self.button_box.border_width_set(15)
        self.button_box.show()

        self._vbox.append(self.button_box, etk.VBox.START, etk.VBox.FILL, 0)

        self._vbox.show()
        self.add(self._vbox)

        self.on_key_down(self._key_down_cb)
        self.on_key_up(self._key_up_cb)

    def _key_down_cb(self, o, ev):
        if self.handle_key_down:
            return self.handle_key_down(ev)

    def _key_up_cb(self, o, ev):
        if self.handle_key_up:
            return self.handle_key_up(ev)

    def _child_get(self):
        return None

    def header_area_append(self, widget):
        "Appends a widget into the header area"
        self.header_area.append(widget, etk.HBox.START, etk.HBox.EXPAND_FILL, 0)
        self.header_area.show()

    def button_add(self, label, show=True):
        """Adds a button in the Button Box.

        First button goes to the right side (end of HBox) while others go
        left side (start of HBox).
        """
        button = etk.Button(label=label)
        button.focusable = False
        button.on_clicked(self._on_button_clicked)
        if show:
            button.show()

        if not self.button_box.children:
            self.button_box.append(button, etk.HBox.END, etk.HBox.FILL, 0)
        else:
            self.button_box.append(button, etk.HBox.START, etk.HBox.FILL, 0)

        return button

    def _on_button_clicked(self, button):
        log.debug("Button %s clicked", button.label_get())


class PanelContentList(PanelContent):
    """Basic panel content with a List in the center.

    You must subclass and define L{model_get()} and L{columns_get()} functions
    to fill the list.
    """
    def __init__(self, evas_obj):
        self.list = None
        PanelContent.__init__(self, evas_obj)

    def _child_get(self):
        return self._create_list()

    def _create_list(self):
        if not self.list:
            columns = self.columns_get()
            model = self.model_get()

            if not model:
                model = etk.ListModel()

            self.list = etk.List(model=model,
                                 selectable=etk.List.NOT_SELECTABLE,
                                 columns=columns)
            self.list.focusable = False
        return self.list

    def model_get(self):
        return None

    def columns_get(self):
        raise NotImplementedError("%s.columns_get() not implemented." % \
                                      self.__class__.__name__)

    def item_append(self, *args):
        self.list.model.append(*args)


class PanelContentFrame(PanelContent):
    def __init__(self, evas_obj):
        self.frame = None
        PanelContent.__init__(self, evas_obj)

    def _child_get(self):
        self.frame = etk.Frame()
        return self.frame


class PanelRich(Panel):
    def __init__(self, main_window, title, child=None, theme=None):
        Panel.__init__(self, main_window, title, theme)

        self.callback_escape = None

        self._init_embed(child)
        self.contents_set(self.embed.object)

    def _init_embed(self, child=None):
        self.embed = etk.Embed(self.evas)
        self._vbox = etk.VBox()

        self.header_area = etk.HBox()
        self.header_area.border_width_set(15)
        self._vbox.append(self.header_area, etk.VBox.START, etk.VBox.FILL, 0)

        self._child = child
        if self._child:
            self._child.show()
            self._vbox.append(self._child, etk.VBox.START,
                              etk.VBox.EXPAND_FILL, 0)

        self.button_box = etk.HBox()
        self.button_box.border_width_set(15)
        self.button_box.show()
        self._vbox.append(self.button_box, etk.VBox.START, etk.VBox.FILL, 0)

        self._vbox.show()
        self.embed.add(self._vbox)

        self.embed.on_key_down(self._key_down_cb)
        self.embed.on_key_up(self._key_up_cb)

    def _key_down_cb(self, o, ev):
        return self.handle_key_down(ev)

    def _key_up_cb(self, o, ev):
        return self.handle_key_up(ev)

    def handle_key_down(self, ev):
        if ev.key == "Escape":
            if self.callback_escape:
                self.callback_escape()
            return False
        return True

    def handle_key_up(self, ev):
        return True

    def header_area_append(self, widget):
        """Appends a widget into the header area."""
        self.header_area.append(widget, etk.HBox.START,
                                etk.HBox.EXPAND_FILL, 0)
        self.header_area.show()

    def button_add(self, label, func=None, show=True):
        """Adds a button in the Button Box.

        @param label: text label in the button.
        @param func: function to be executed when button is clicked.
        @param show: whether the button starts visible or not.

        First button goes to the right side (end of HBox) while others go
        left side (start of HBox).

        """
        button = etk.Button(label=label)
        button.focusable = False
        button.on_clicked(func)
        if show:
            button.show()

        if not self.button_box.children:
            self.button_box.append(button, etk.HBox.END, etk.HBox.FILL, 0)
        else:
            self.button_box.append(button, etk.HBox.START, etk.HBox.FILL, 0)

        return button

    @evas.decorators.del_callback
    def _destroy_contents(self):
        self.embed.destroy()

    @evas.decorators.focus_in_callback
    def _focus_in(self):
        self.embed.object.focus = True
        self.do_on_focus()

    def do_on_focus(self):
        pass


class PanelRichList(PanelRich):
    def __init__(self, main_window, title, elements, theme=None):
        self._create_list(elements)
        PanelRich.__init__(self, main_window, title, self.list, theme)

    def list_columns_get(self):
        raise NotImplementedError("%s.list_columns_get() not implemented." % \
                                      self.__class__.__name__)

    def list_model_get(self, elements):
        if isinstance(elements, etk.ListModelMixin):
            return elements
        else:
            return etk.ListModel()

    def _create_list(self, elements):
        columns = self.list_columns_get()
        self.list_model = self.list_model_get(elements)

        self.list = etk.List(model=self.list_model,
                             selectable=etk.List.NOT_SELECTABLE,
                             columns=columns)
        self.list.focusable = False

    def list_item_append(self, *args):
        self.list.model.append(*args)
