/**
 * scim_panel_vkb_setup.cpp:implementation of
 * Setup Module of scim-panel-vkb-gtk.
 * 
 * Copyright (c) 2002-2005 James Su <suzhe@tsinghua.org.cn>
 * Copyright (C) 2009, Intel Corporation.
 *
 * Author: Vincent Huang  <chenglan.huang@intel.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA  02111-1307  USA
 */

#define Uses_SCIM_CONFIG_BASE
#define Uses_SCIM_PANEL

#include <gtk/gtk.h>
#include <sys/stat.h>
#include "scim.h"

#include "config.h"
#if defined(ENABLE_NLS)
#include <glib/gi18n-lib.h>
#else
#define _(String) (String)
#define N_(String) (String)
#define bindtextdomain(Package,Directory)
#define bind_textdomain_codeset(domain,codeset)
#endif

using namespace scim;

#define scim_module_init panel_vkb_setup_LTX_scim_module_init
#define scim_module_exit panel_vkb_setup_LTX_scim_module_exit

#define scim_setup_module_create_ui       panel_vkb_setup_LTX_scim_setup_module_create_ui
#define scim_setup_module_get_category    panel_vkb_setup_LTX_scim_setup_module_get_category
#define scim_setup_module_get_name        panel_vkb_setup_LTX_scim_setup_module_get_name
#define scim_setup_module_get_description panel_vkb_setup_LTX_scim_setup_module_get_description
#define scim_setup_module_load_config     panel_vkb_setup_LTX_scim_setup_module_load_config
#define scim_setup_module_save_config     panel_vkb_setup_LTX_scim_setup_module_save_config
#define scim_setup_module_query_changed   panel_vkb_setup_LTX_scim_setup_module_query_changed

#define SCIM_CONFIG_PANEL_VKB_GTK_FONT                      "/Panel/VKB/Gtk/Font"
#define SCIM_CONFIG_PANEL_VKB_GTK_TOOLBAR_ALWAYS_SHOW       "/Panel/VKB/Gtk/ToolBar/AlwaysShow"
#define SCIM_CONFIG_PANEL_VKB_GTK_TOOLBAR_ALWAYS_HIDDEN     "/Panel/VKB/Gtk/ToolBar/AlwaysHidden"
#define SCIM_CONFIG_PANEL_VKB_GTK_TOOLBAR_AUTO_SNAP         "/Panel/VKB/Gtk/ToolBar/AutoSnap"
#define SCIM_CONFIG_PANEL_VKB_GTK_TOOLBAR_HIDE_TIMEOUT      "/Panel/VKB/Gtk/ToolBar/HideTimeout"
#define SCIM_CONFIG_PANEL_VKB_GTK_TOOLBAR_POS_X             "/Panel/VKB/Gtk/ToolBar/POS_X"
#define SCIM_CONFIG_PANEL_VKB_GTK_TOOLBAR_POS_Y             "/Panel/VKB/Gtk/ToolBar/POS_Y"
#define SCIM_CONFIG_PANEL_VKB_GTK_TOOLBAR_SHOW_FACTORY_ICON "/Panel/VKB/Gtk/ToolBar/ShowFactoryIcon"
#define SCIM_CONFIG_PANEL_VKB_GTK_TOOLBAR_SHOW_FACTORY_NAME "/Panel/VKB/Gtk/ToolBar/ShowFactoryName"
#define SCIM_CONFIG_PANEL_VKB_GTK_TOOLBAR_SHOW_STICK_ICON   "/Panel/VKB/Gtk/ToolBar/ShowStickIcon"
#define SCIM_CONFIG_PANEL_VKB_GTK_TOOLBAR_SHOW_MENU_ICON    "/Panel/VKB/Gtk/ToolBar/ShowMenuIcon"
#define SCIM_CONFIG_PANEL_VKB_GTK_TOOLBAR_SHOW_HELP_ICON    "/Panel/VKB/Gtk/ToolBar/ShowHelpIcon"
#define SCIM_CONFIG_PANEL_VKB_GTK_TOOLBAR_SHOW_PROPERTY_LABEL "/Panel/VKB/Gtk/ToolBar/ShowPropertyLabel"
#define SCIM_CONFIG_PANEL_VKB_GTK_DEFAULT_STICKED           "/Panel/VKB/Gtk/DefaultSticked"
#define SCIM_CONFIG_PANEL_VKB_GTK_SHOW_TRAY_ICON            "/Panel/VKB/Gtk/ShowTrayIcon"
#define SCIM_CONFIG_PANEL_VKB_GTK_DOCK_PANEL                "/Panel/VKB/Gtk/DockPanel"
#define SCIM_CONFIG_PANEL_VKB_GTK_PANEL_EXTEND              "/Panel/VKB/Gtk/PanelExtend"
#define SCIM_CONFIG_PANEL_VKB_GTK_VKB_PROGRAM               "/Panel/VKB/Gtk/VKBProgram"
#define SCIM_CONFIG_PANEL_VKB_GTK_VKB_PARAMETERS            "/Panel/VKB/Gtk/VKBParameters"
#define SCIM_CONFIG_PANEL_VKB_GTK_VKB_WIDTH                 "/Panel/VKB/Gtk/VKB_Width"
#define SCIM_CONFIG_PANEL_VKB_GTK_VKB_HEIGHT                "/Panel/VKB/Gtk/VKB_Height"
#define SCIM_DEFAULT_VKB_PROGRAM                            "fvkbd-gtk"
#define SCIM_DEFAULT_VKB_PARAMETERS                         "--xid"
#define SCIM_DEFAULT_VKB_WINDOW_WIDTH                       800
#define SCIM_DEFAULT_VKB_WINDOW_HEIGHT                      250
#define SCIM_DEFAULT_VKB_DOCKPANEL                          true
#define SCIM_DEFAULT_VKB_PANEL_EXTEND                       true


static GtkWidget * create_setup_window ();
static void        load_config (const ConfigPointer &config);
static void        save_config (const ConfigPointer &config);
static bool        query_changed ();

// Module Interface.
extern "C" {
    void scim_module_init (void)
    {
        bindtextdomain (GETTEXT_PACKAGE, LOCALEDIR);
        bind_textdomain_codeset(GETTEXT_PACKAGE, "UTF-8");
    }

    void scim_module_exit (void)
    {
    }

    GtkWidget * scim_setup_module_create_ui (void)
    {
        return create_setup_window ();
    }

    String scim_setup_module_get_category (void)
    {
        return String ("Panel");
    }

    String scim_setup_module_get_name (void)
    {
        return String (_("Virtual Keyboard"));
    }

    String scim_setup_module_get_description (void)
    {
        return String (_("A panel daemon with VKB."));
    }

    void scim_setup_module_load_config (const ConfigPointer &config)
    {
        load_config (config);
    }

    void scim_setup_module_save_config (const ConfigPointer &config)
    {
        save_config (config);
    }

    bool scim_setup_module_query_changed ()
    {
        return query_changed ();
    }
} // extern "C"

// Internal data declaration.
static bool   __config_toolbar_always_show       = false;
static bool   __config_toolbar_always_hidden     = false;
static bool   __config_toolbar_auto_snap         = false;
static int    __config_toolbar_hide_timeout      = 2;
static bool   __config_toolbar_show_factory_icon = true;
static bool   __config_toolbar_show_factory_name = true;
static bool   __config_toolbar_show_stick_icon   = false;
static bool   __config_toolbar_show_menu_icon    = true;
static bool   __config_toolbar_show_help_icon    = false;
static bool   __config_toolbar_show_property_label = true;
static bool   __config_default_sticked           = false;
static bool   __config_show_tray_icon            = true;
static bool   __config_vkb_dockpanel             = bool (SCIM_DEFAULT_VKB_DOCKPANEL);
static bool   __config_vkb_panel_extend          = bool (SCIM_DEFAULT_VKB_PANEL_EXTEND);
static gint   __config_vkb_window_width          = gint (SCIM_DEFAULT_VKB_WINDOW_WIDTH);
static gint   __config_vkb_window_height         = gint (SCIM_DEFAULT_VKB_WINDOW_HEIGHT);

static String __config_font                      = _("sans 16");
static String __config_vkb_program               = String (SCIM_DEFAULT_VKB_PROGRAM);
static String __config_vkb_parameters            = String (SCIM_DEFAULT_VKB_PARAMETERS);

static bool   __have_changed                     = false;
static gint   __default_screen_width             = gint (SCIM_DEFAULT_VKB_WINDOW_WIDTH);
static gint   __default_screen_height            = gint (SCIM_DEFAULT_VKB_WINDOW_HEIGHT);

static GtkWidget * __widget_toolbar_show_behaviour    = 0;
static GtkWidget * __widget_toolbar_hide_timeout      = 0;
static GtkWidget * __widget_toolbar_show_factory_icon  = 0;
static GtkWidget * __widget_toolbar_show_factory_name  = 0;
static GtkWidget * __widget_toolbar_show_stick_icon   = 0;
static GtkWidget * __widget_toolbar_show_menu_icon   = 0;
static GtkWidget * __widget_toolbar_show_help_icon    = 0;
static GtkWidget * __widget_toolbar_show_property_label = 0;
static GtkWidget * __widget_default_sticked           = 0;
static GtkWidget * __widget_show_tray_icon            = 0;
static GtkWidget * __widget_font                      = 0;
static GtkWidget * __widget_vkb_program               = 0;
static GtkWidget * __widget_vkb_program_selection     = 0;
static GtkWidget * __widget_vkb_parameters            = 0;
static GtkWidget * __widget_vkb_dockpanel             = 0;
static GtkWidget * __widget_vkb_panel_extend          = 0;
static GtkWidget * __widget_vkb_window_height         = 0;
static GtkWidget * __widget_vkb_window_width          = 0;
static GtkWidget * __widget_vkb_set_default           = 0;

static GtkTooltips * __widget_tooltips                = 0;

enum ToolbarShowFlavourType {
    SCIM_TOOLBAR_SHOW_ALWAYS,
    SCIM_TOOLBAR_SHOW_ON_DEMAND,
    SCIM_TOOLBAR_SHOW_NEVER
};

static const char * __toolbar_show_behaviour_text[] = {
    _("Always"),
    _("On demand"),
    _("Never")
};

// Declaration of internal functions.
static void
on_default_toggle_button_toggled     (GtkToggleButton *togglebutton,
                                      gpointer         user_data);

static void
on_default_spin_button_changed       (GtkSpinButton   *spinbutton,
                                      gpointer         user_data);

static void
on_toolbar_show_behaviour_changed      (GtkComboBox     *combobox,
                                      gpointer         user_data);

static void
on_font_selection_clicked            (GtkButton       *button,
                                      gpointer         user_data);

static void
on_vkb_program_changed               (GtkEntry         *entry,
                                      gpointer         user_data);

static void
on_vkb_program_selection_clicked     (GtkButton *button,
                                      gpointer   user_data);

static void
on_vkb_parameters_changed            (GtkEntry         *entry,
                                      gpointer         user_data);

static void
on_vkb_set_default                   (GtkButton *button,
                                      gpointer   user_data);

static void
setup_widget_value ();

// Function implementations.
GtkWidget *
create_setup_window ()
{
    static GtkWidget *window = 0;

    GdkScreen* screen = NULL;

    screen = gdk_screen_get_default();
    __default_screen_width = gdk_screen_get_width(screen);
    __default_screen_height = gdk_screen_get_height(screen);

    if (!window) {
        GtkWidget *notebook;
        GtkWidget *table;
        GtkWidget *frame;
        GtkWidget *vbox;
        GtkWidget *label;
        GtkWidget *hbox;

        __widget_tooltips = gtk_tooltips_new ();
        // Create the Notebook.
        notebook = gtk_notebook_new ();
        gtk_widget_show (notebook);

        // Create the Input Method panel setup page.
        vbox = gtk_vbox_new (FALSE, 0);
        gtk_widget_show (vbox);
        gtk_container_add (GTK_CONTAINER (notebook), vbox);

        // Create the label for this note page.
        label = gtk_label_new (_("IM Panel"));
        gtk_widget_show (label);
        gtk_notebook_set_tab_label (GTK_NOTEBOOK (notebook), gtk_notebook_get_nth_page (GTK_NOTEBOOK (notebook), 0), label);

        // Create the ToolBar setup block.
        frame = gtk_frame_new (_("ToolBar"));
        gtk_widget_show (frame);
        gtk_box_pack_start (GTK_BOX (vbox), frame, FALSE, FALSE, 0);
        gtk_container_set_border_width (GTK_CONTAINER (frame), 4);

        table = gtk_table_new (4, 2, FALSE);
        gtk_widget_show (table);
        gtk_container_add (GTK_CONTAINER (frame), table);
        gtk_table_set_row_spacings (GTK_TABLE (table), 4);
        gtk_table_set_col_spacings (GTK_TABLE (table), 8);

        hbox = gtk_hbox_new (FALSE, 0);
        gtk_widget_show (hbox);

        gtk_table_attach (GTK_TABLE (table), hbox, 0, 1, 0, 1,
                (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
                (GtkAttachOptions) (GTK_EXPAND), 4, 0);

        label = gtk_label_new_with_mnemonic (_("Show:"));
        gtk_widget_show (label);
        gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);
        gtk_misc_set_padding (GTK_MISC (label), 4, 0);

        __widget_toolbar_show_behaviour = gtk_combo_box_new_text ();
        gtk_combo_box_append_text (GTK_COMBO_BOX (__widget_toolbar_show_behaviour), 
                (__toolbar_show_behaviour_text[SCIM_TOOLBAR_SHOW_ALWAYS]));
        gtk_combo_box_append_text (GTK_COMBO_BOX (__widget_toolbar_show_behaviour),
                (__toolbar_show_behaviour_text[SCIM_TOOLBAR_SHOW_ON_DEMAND]));
        gtk_combo_box_append_text (GTK_COMBO_BOX (__widget_toolbar_show_behaviour),
                (__toolbar_show_behaviour_text[SCIM_TOOLBAR_SHOW_NEVER]));
        gtk_widget_show (__widget_toolbar_show_behaviour);
        gtk_box_pack_start (GTK_BOX (hbox), __widget_toolbar_show_behaviour, FALSE, FALSE, 0);
        gtk_label_set_mnemonic_widget (GTK_LABEL (label), __widget_toolbar_show_behaviour);

        __widget_toolbar_show_factory_icon = gtk_check_button_new_with_mnemonic (
                                             _("Show _input method icon"));
        gtk_widget_show (__widget_toolbar_show_factory_icon);
        gtk_table_attach (GTK_TABLE (table), __widget_toolbar_show_factory_icon, 0, 1, 1, 2,
                (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
                (GtkAttachOptions) (GTK_EXPAND), 4, 0);

        __widget_toolbar_show_factory_name = gtk_check_button_new_with_mnemonic (
                                             _("Show inp_ut method name"));
        gtk_widget_show (__widget_toolbar_show_factory_name);
        gtk_table_attach (GTK_TABLE (table), __widget_toolbar_show_factory_name, 0, 1, 2, 3,
                (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
                (GtkAttachOptions) (GTK_EXPAND), 4, 0);

        hbox = gtk_hbox_new (FALSE, 0);
        gtk_widget_show (hbox);
        gtk_table_attach (GTK_TABLE (table), hbox, 1, 2, 0, 1,
                (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
                (GtkAttachOptions) (GTK_EXPAND), 4, 0);

        label = gtk_label_new_with_mnemonic (_("Hi_de timeout:"));
        gtk_widget_show (label);
        gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);
        gtk_misc_set_padding (GTK_MISC (label), 4, 0);

        __widget_toolbar_hide_timeout = gtk_spin_button_new_with_range (0, 60, 1);
        gtk_widget_show (__widget_toolbar_hide_timeout);
        gtk_box_pack_start (GTK_BOX (hbox), __widget_toolbar_hide_timeout, FALSE, FALSE, 0);
        gtk_spin_button_set_numeric (GTK_SPIN_BUTTON (__widget_toolbar_hide_timeout), TRUE);
        gtk_spin_button_set_snap_to_ticks (GTK_SPIN_BUTTON (__widget_toolbar_hide_timeout), TRUE);
        gtk_spin_button_set_digits (GTK_SPIN_BUTTON (__widget_toolbar_hide_timeout), 0);
        gtk_label_set_mnemonic_widget (GTK_LABEL (label), __widget_toolbar_hide_timeout);

        __widget_toolbar_show_stick_icon = gtk_check_button_new_with_mnemonic (_("Show s_tick icon"));
        gtk_widget_show (__widget_toolbar_show_stick_icon);
        gtk_table_attach (GTK_TABLE (table), __widget_toolbar_show_stick_icon, 1, 2, 1, 2,
                (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
                (GtkAttachOptions) (GTK_EXPAND), 4, 0);

        __widget_toolbar_show_menu_icon = gtk_check_button_new_with_mnemonic (_("Show m_enu icon"));
        gtk_widget_show (__widget_toolbar_show_menu_icon);
        gtk_table_attach (GTK_TABLE (table), __widget_toolbar_show_menu_icon, 1, 2, 2, 3,
                (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
                (GtkAttachOptions) (GTK_EXPAND), 4, 0);

        __widget_toolbar_show_help_icon = gtk_check_button_new_with_mnemonic (_("Show _help icon"));
        gtk_widget_show (__widget_toolbar_show_help_icon);
        gtk_table_attach (GTK_TABLE (table), __widget_toolbar_show_help_icon, 1, 2, 3, 4,
                (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
                (GtkAttachOptions) (GTK_EXPAND), 4, 0);

        __widget_toolbar_show_property_label = gtk_check_button_new_with_mnemonic (_("Show _property label"));
        gtk_widget_show (__widget_toolbar_show_property_label);
        gtk_table_attach (GTK_TABLE (table), __widget_toolbar_show_property_label, 0, 1, 3, 4,
                (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
                (GtkAttachOptions) (GTK_EXPAND), 4, 0);

        hbox = gtk_hbox_new (FALSE, 8);
        gtk_widget_show (hbox);
        gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);

        frame = gtk_frame_new (_("Misc"));
        gtk_widget_show (frame);
        gtk_container_set_border_width (GTK_CONTAINER (frame), 4);
        gtk_box_pack_start (GTK_BOX (hbox), frame, TRUE, TRUE, 0);

        vbox = gtk_vbox_new (FALSE, 4);
        gtk_widget_show (vbox);
        gtk_container_add (GTK_CONTAINER (frame), vbox);

        __widget_show_tray_icon = gtk_check_button_new_with_mnemonic (_("Show tra_y icon"));
        gtk_widget_show (__widget_show_tray_icon);
        gtk_box_pack_start (GTK_BOX (vbox), __widget_show_tray_icon, FALSE, FALSE, 0);

        __widget_default_sticked = gtk_check_button_new_with_mnemonic (_("Stick _windows"));
        gtk_widget_show (__widget_default_sticked);
        gtk_box_pack_start (GTK_BOX (vbox), __widget_default_sticked, FALSE, FALSE, 0);

        hbox = gtk_hbox_new (FALSE, 0);
        gtk_widget_show (hbox);
        gtk_box_pack_start (GTK_BOX (vbox), hbox, TRUE, TRUE, 0);

        label = gtk_label_new_with_mnemonic (_("_Font:"));
        gtk_widget_show (label);
        gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);
        gtk_misc_set_padding (GTK_MISC (label), 4, 0);

        __widget_font = gtk_button_new_with_label (_("default"));
        gtk_widget_show (__widget_font);
        gtk_container_set_border_width (GTK_CONTAINER (__widget_font), 4);
        gtk_box_pack_start (GTK_BOX (hbox), __widget_font, FALSE, FALSE, 0);
        gtk_label_set_mnemonic_widget (GTK_LABEL (label), __widget_font);


        // Create the Virtual Keyboard setup page.
        vbox = gtk_vbox_new (FALSE, 0);
        gtk_widget_show (vbox);
        gtk_container_add (GTK_CONTAINER (notebook), vbox);

        // Create the label for this note page.
        label = gtk_label_new (_("VKB"));
        gtk_widget_show (label);
        gtk_notebook_set_tab_label (GTK_NOTEBOOK (notebook),
                gtk_notebook_get_nth_page (GTK_NOTEBOOK (notebook), 1), label);

        frame = gtk_frame_new (_("Options"));
        gtk_widget_show (frame);
        gtk_container_set_border_width (GTK_CONTAINER (frame), 4);
        gtk_box_pack_start (GTK_BOX (vbox), frame, FALSE, FALSE, 0);

        table = gtk_table_new (2, 2, FALSE);
        gtk_widget_show (table);
        gtk_container_add (GTK_CONTAINER (frame), table);
        gtk_table_set_row_spacings (GTK_TABLE (table), 4);
        gtk_table_set_col_spacings (GTK_TABLE (table), 1);

        label = gtk_label_new_with_mnemonic (_("VKB _Program:"));
        gtk_widget_show (label);
        gtk_table_attach (GTK_TABLE (table), label, 0, 1, 0, 1,
                (GtkAttachOptions) (0),
                (GtkAttachOptions) (GTK_EXPAND), 4, 0);

        hbox = gtk_hbox_new(FALSE, 0);
        gtk_widget_show (hbox);
        gtk_table_attach (GTK_TABLE (table), hbox,1, 2, 0, 1,
                (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
                (GtkAttachOptions) (GTK_EXPAND), 4, 0);

        __widget_vkb_program = gtk_entry_new ();
        gtk_widget_show(__widget_vkb_program);
        gtk_box_pack_start(GTK_BOX(hbox), __widget_vkb_program, TRUE, TRUE,0);

        __widget_vkb_program_selection = gtk_button_new_with_label ("...");
        gtk_widget_show(__widget_vkb_program_selection);
        gtk_box_pack_start(GTK_BOX(hbox), __widget_vkb_program_selection ,
                           FALSE, FALSE,2);

        label = gtk_label_new_with_mnemonic (_("Para_meters:"));
        gtk_widget_show (label);
        gtk_table_attach (GTK_TABLE (table), label, 0, 1, 1, 2,
                          (GtkAttachOptions) (0),
                          (GtkAttachOptions) (GTK_EXPAND), 4, 0);

        __widget_vkb_parameters = gtk_entry_new();
        gtk_widget_show(__widget_vkb_parameters);
        gtk_table_attach (GTK_TABLE (table), __widget_vkb_parameters,
                          1, 2, 1, 2,
                          (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
                          (GtkAttachOptions) (GTK_EXPAND), 4, 0);

        frame = gtk_frame_new (_("Layout"));
        gtk_widget_show (frame);
        gtk_container_set_border_width (GTK_CONTAINER (frame), 4);
        gtk_box_pack_start (GTK_BOX (vbox), frame, TRUE, TRUE, 0);

        table = gtk_table_new(3, 2, FALSE);
        gtk_widget_show(table);
        gtk_container_add (GTK_CONTAINER (frame), table);
        gtk_table_set_row_spacings (GTK_TABLE (table), 4);
        gtk_table_set_col_spacings (GTK_TABLE (table), 8);

        hbox = gtk_hbox_new (FALSE, 0);
        gtk_widget_show (hbox);
        gtk_table_attach (GTK_TABLE (table), hbox, 0, 1, 0, 1,
                (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
                (GtkAttachOptions) (GTK_EXPAND), 4, 0);

        label = gtk_label_new_with_mnemonic (_("_Width:"));
        gtk_widget_show (label);
        gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);
        gtk_misc_set_padding (GTK_MISC (label), 4, 0);

        __widget_vkb_window_width = gtk_spin_button_new_with_range(0, __default_screen_width, 1);
        gtk_widget_show(__widget_vkb_window_width);
        gtk_box_pack_start (GTK_BOX (hbox), __widget_vkb_window_width, FALSE, FALSE, 0);
        gtk_spin_button_set_numeric (GTK_SPIN_BUTTON (__widget_vkb_window_width), TRUE);
        gtk_spin_button_set_snap_to_ticks (GTK_SPIN_BUTTON (__widget_vkb_window_width), TRUE);
        gtk_spin_button_set_digits (GTK_SPIN_BUTTON (__widget_vkb_window_width), 0);
        gtk_label_set_mnemonic_widget (GTK_LABEL (label), __widget_vkb_window_width);


        hbox = gtk_hbox_new (FALSE, 0);
        gtk_widget_show (hbox);
        gtk_table_attach (GTK_TABLE (table), hbox, 1, 2, 0, 1,
                (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
                (GtkAttachOptions) (GTK_EXPAND), 4, 0);

        label = gtk_label_new_with_mnemonic (_("_Height:"));
        gtk_widget_show (label);
        gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);
        gtk_misc_set_padding (GTK_MISC (label), 4, 0);

        __widget_vkb_window_height = gtk_spin_button_new_with_range(0,
                                     __default_screen_height, 1);
        gtk_widget_show(__widget_vkb_window_height);
        gtk_box_pack_start (GTK_BOX (hbox), __widget_vkb_window_height, FALSE, FALSE, 0);
        gtk_spin_button_set_numeric (GTK_SPIN_BUTTON (__widget_vkb_window_height), TRUE);
        gtk_spin_button_set_snap_to_ticks (GTK_SPIN_BUTTON (__widget_vkb_window_height), TRUE);
        gtk_spin_button_set_digits (GTK_SPIN_BUTTON (__widget_vkb_window_height), 0);
        gtk_label_set_mnemonic_widget (GTK_LABEL (label), __widget_vkb_window_height);

        __widget_vkb_dockpanel = gtk_check_button_new_with_mnemonic (_("_Dock panel"));
        gtk_widget_show (__widget_vkb_dockpanel);
        gtk_table_attach (GTK_TABLE (table), __widget_vkb_dockpanel, 0, 1, 1, 2,
                (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
                (GtkAttachOptions) (GTK_EXPAND), 4, 0);

        __widget_vkb_panel_extend = gtk_check_button_new_with_mnemonic (
                _("_Extended VKB Panel"));
        gtk_widget_show (__widget_vkb_panel_extend);
        gtk_table_attach (GTK_TABLE (table), __widget_vkb_panel_extend, 1, 2, 1, 2,
                (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
                (GtkAttachOptions) (GTK_EXPAND), 4, 0);

        __widget_vkb_set_default = gtk_button_new_with_mnemonic (_("Reset Default _Value"));
        gtk_widget_show(__widget_vkb_set_default);
        gtk_table_attach (GTK_TABLE (table), __widget_vkb_set_default, 0, 1, 2, 3,
                (GtkAttachOptions) (GTK_EXPAND),
                (GtkAttachOptions) (GTK_EXPAND), 4, 0);


        // Connect all signals.
        g_signal_connect ((gpointer) __widget_toolbar_show_behaviour, "changed",
                G_CALLBACK (on_toolbar_show_behaviour_changed),
                NULL);

        g_signal_connect ((gpointer) __widget_toolbar_hide_timeout, "value_changed",
                G_CALLBACK (on_default_spin_button_changed),
                &__config_toolbar_hide_timeout);

        g_signal_connect ((gpointer) __widget_toolbar_show_factory_icon, "toggled",
                G_CALLBACK (on_default_toggle_button_toggled),
                &__config_toolbar_show_factory_icon);

        g_signal_connect ((gpointer) __widget_toolbar_show_factory_name, "toggled",
                G_CALLBACK (on_default_toggle_button_toggled),
                &__config_toolbar_show_factory_name);

        g_signal_connect ((gpointer) __widget_toolbar_show_stick_icon, "toggled",
                G_CALLBACK (on_default_toggle_button_toggled),
                &__config_toolbar_show_stick_icon);

        g_signal_connect ((gpointer) __widget_toolbar_show_menu_icon, "toggled",
                G_CALLBACK (on_default_toggle_button_toggled),
                &__config_toolbar_show_menu_icon);

        g_signal_connect ((gpointer) __widget_toolbar_show_help_icon, "toggled",
                G_CALLBACK (on_default_toggle_button_toggled),
                &__config_toolbar_show_help_icon);

        g_signal_connect ((gpointer) __widget_toolbar_show_property_label, "toggled",
                G_CALLBACK (on_default_toggle_button_toggled),
                &__config_toolbar_show_property_label);

        g_signal_connect ((gpointer) __widget_default_sticked, "toggled",
                G_CALLBACK (on_default_toggle_button_toggled),
                &__config_default_sticked);

        g_signal_connect ((gpointer) __widget_show_tray_icon, "toggled",
                G_CALLBACK (on_default_toggle_button_toggled),
                &__config_show_tray_icon);

        g_signal_connect ((gpointer) __widget_font, "clicked",
                G_CALLBACK (on_font_selection_clicked),
                NULL);
    
        //for VKB
        g_signal_connect ((gpointer) __widget_vkb_program, "changed",
                G_CALLBACK (on_vkb_program_changed),
                NULL);

        g_signal_connect ((gpointer) __widget_vkb_program_selection, "clicked",
                G_CALLBACK (on_vkb_program_selection_clicked),
                NULL);

        g_signal_connect ((gpointer) __widget_vkb_parameters, "changed",
                G_CALLBACK (on_vkb_parameters_changed),
                NULL);

        g_signal_connect ((gpointer) __widget_vkb_window_width, "value_changed",
                G_CALLBACK (on_default_spin_button_changed),
                &__config_vkb_window_width);

        g_signal_connect ((gpointer) __widget_vkb_window_height, "value_changed",
                G_CALLBACK (on_default_spin_button_changed),
                &__config_vkb_window_height);

        g_signal_connect ((gpointer) __widget_vkb_dockpanel, "toggled",
                G_CALLBACK (on_default_toggle_button_toggled),
                &__config_vkb_dockpanel);

        g_signal_connect ((gpointer) __widget_vkb_panel_extend, "toggled",
                G_CALLBACK (on_default_toggle_button_toggled),
                &__config_vkb_panel_extend);

        g_signal_connect ((gpointer) __widget_vkb_set_default, "clicked",
                G_CALLBACK (on_vkb_set_default),
                NULL);

        // Set all tooltips.
        gtk_tooltips_set_tip (__widget_tooltips, __widget_toolbar_show_behaviour,
               _("If option \"Always\" is selected, "
                 "the toolbar will always be shown on the screen. "
                 "If option \"On demand\" is selected, it will only be shown when SCIM "
                 "is activated. "
                 "If option \"Never\" is selected, it will never be shown."), NULL);

        gtk_tooltips_set_tip (__widget_tooltips, __widget_toolbar_hide_timeout,
               _("The toolbar will be hidden out after "
                 "this timeout is elapsed. "
                 "This option is only valid when "
                 "\"Always show\" is selected. "
                 "Set to zero to disable this behavior."), NULL);

        gtk_tooltips_set_tip (__widget_tooltips, __widget_toolbar_show_factory_icon,
               _("If this option is checked, "
                 "the input method icon will be showed on the toolbar."), NULL);

        gtk_tooltips_set_tip (__widget_tooltips, __widget_toolbar_show_factory_name,
               _("If this option is checked, "
                 "the input method name will be showed on the toolbar."), NULL);

        gtk_tooltips_set_tip (__widget_tooltips, __widget_toolbar_show_stick_icon,
               _("If this option is checked, "
                 "the stick icon will be showed on the toolbar."), NULL);

        gtk_tooltips_set_tip (__widget_tooltips, __widget_toolbar_show_menu_icon,
               _("If this option is checked, "
                 "the menu icon will be showed on the toolbar."), NULL);

        gtk_tooltips_set_tip (__widget_tooltips, __widget_toolbar_show_help_icon,
               _("If this option is checked, "
                 "the help icon will be showed on the toolbar."), NULL);

        gtk_tooltips_set_tip (__widget_tooltips, __widget_toolbar_show_property_label,
               _("If this option is checked, "
                 "the text label of input method properties will be showed on the toolbar."), NULL);

        gtk_tooltips_set_tip (__widget_tooltips, __widget_show_tray_icon,
               _("If this option is checked, "
                 "the tray icon will be showed on the desktop's taskbar."), NULL);

        gtk_tooltips_set_tip (__widget_tooltips, __widget_default_sticked,
               _("If this option is checked, "
                 "the toolbar, input and lookup table "
                 "windows will be sticked to "
                 "its original position."),NULL);

        gtk_tooltips_set_tip (__widget_tooltips, __widget_font,
               _("The font setting will be used in "
                 "the input and lookup table windows."), NULL);

        gtk_tooltips_set_tip (__widget_tooltips, __widget_vkb_program,
               _("Set the virtual keyboard program" ), NULL);

        gtk_tooltips_set_tip (__widget_tooltips, __widget_vkb_parameters,
               _("Set runtime parameters for the virtual keyboard program"), NULL);

        gtk_tooltips_set_tip (__widget_tooltips, __widget_vkb_window_width,
               _("Set vkb panel's width"), NULL);

        gtk_tooltips_set_tip (__widget_tooltips, __widget_vkb_window_height,
               _("Set vkb panel's height"), NULL);

        gtk_tooltips_set_tip (__widget_tooltips, __widget_vkb_dockpanel,
               _("If this option is checked, "
                 "the vkb panel will always be showed on the bottom of screen."), NULL);

        gtk_tooltips_set_tip (__widget_tooltips, __widget_vkb_panel_extend,
               _("If this option is checked, "
                 "the vkb panel will extend to the screen width no matter what window width you set."), NULL);

        gtk_tooltips_set_tip (__widget_tooltips, __widget_vkb_set_default,
               _("Reset all the options to default value "), NULL);

        window = notebook;

        setup_widget_value ();
    }
    return window;
}

    void
setup_widget_value ()
{
    if (__widget_toolbar_show_behaviour) {
        if (__config_toolbar_always_hidden) {
            gtk_combo_box_set_active (
                    GTK_COMBO_BOX (__widget_toolbar_show_behaviour),
                    SCIM_TOOLBAR_SHOW_NEVER);
        } else if (__config_toolbar_always_show) {
            gtk_combo_box_set_active (
                    GTK_COMBO_BOX (__widget_toolbar_show_behaviour),
                    SCIM_TOOLBAR_SHOW_ALWAYS);
        } else {
            gtk_combo_box_set_active (
                    GTK_COMBO_BOX (__widget_toolbar_show_behaviour),
                    SCIM_TOOLBAR_SHOW_ON_DEMAND);
        }
    }

    if (__widget_toolbar_hide_timeout) {
        gtk_spin_button_set_value (
                GTK_SPIN_BUTTON (__widget_toolbar_hide_timeout),
                __config_toolbar_hide_timeout);

        gtk_widget_set_sensitive (
                __widget_toolbar_hide_timeout,
                __config_toolbar_always_show);
    }

    if (__widget_toolbar_show_factory_icon) {
        gtk_toggle_button_set_active (
                GTK_TOGGLE_BUTTON (__widget_toolbar_show_factory_icon),
                __config_toolbar_show_factory_icon);
    }

    if (__widget_toolbar_show_factory_name) {
        gtk_toggle_button_set_active (
                GTK_TOGGLE_BUTTON (__widget_toolbar_show_factory_name),
                __config_toolbar_show_factory_name);
    }

    if (__widget_toolbar_show_stick_icon) {
        gtk_toggle_button_set_active (
                GTK_TOGGLE_BUTTON (__widget_toolbar_show_stick_icon),
                __config_toolbar_show_stick_icon);
    }

    if (__widget_toolbar_show_menu_icon) {
        gtk_toggle_button_set_active (
                GTK_TOGGLE_BUTTON (__widget_toolbar_show_menu_icon),
                __config_toolbar_show_menu_icon);
    }

    if (__widget_toolbar_show_help_icon) {
        gtk_toggle_button_set_active (
                GTK_TOGGLE_BUTTON (__widget_toolbar_show_help_icon),
                __config_toolbar_show_help_icon);
    }

    if (__widget_toolbar_show_property_label) {
        gtk_toggle_button_set_active (
                GTK_TOGGLE_BUTTON (__widget_toolbar_show_property_label),
                __config_toolbar_show_property_label);
    }

    if (__widget_default_sticked) {
        gtk_toggle_button_set_active (
                GTK_TOGGLE_BUTTON (__widget_default_sticked),
                __config_default_sticked);
    }

    if (__widget_show_tray_icon) {
        gtk_toggle_button_set_active (
                GTK_TOGGLE_BUTTON (__widget_show_tray_icon),
                __config_show_tray_icon);
    }

    if (__widget_font) {
        gtk_button_set_label (
                GTK_BUTTON (__widget_font),
                __config_font.c_str ());
    }

    if (__widget_vkb_program) {
        gtk_entry_set_text  (
                GTK_ENTRY (__widget_vkb_program),
                __config_vkb_program.c_str());
    }

    if (__widget_vkb_parameters) {
        gtk_entry_set_text  (
                GTK_ENTRY (__widget_vkb_parameters),
                __config_vkb_parameters.c_str());
    }

    if (__widget_vkb_window_width) {
        gtk_spin_button_set_value (
                GTK_SPIN_BUTTON (__widget_vkb_window_width),
                __config_vkb_window_width);
    }

    if (__widget_vkb_window_height) {
        gtk_spin_button_set_value (
                GTK_SPIN_BUTTON (__widget_vkb_window_height),
                __config_vkb_window_height);
    }

    if (__widget_vkb_dockpanel) {
        gtk_toggle_button_set_active (
                GTK_TOGGLE_BUTTON (__widget_vkb_dockpanel),
                __config_vkb_dockpanel);
    }

    if (__widget_vkb_panel_extend) {
        gtk_toggle_button_set_active (
                GTK_TOGGLE_BUTTON (__widget_vkb_panel_extend),
                __config_vkb_panel_extend);
    }

}

    void
load_config (const ConfigPointer &config)
{
    if (!config.null ()) {
        __config_toolbar_always_hidden =
            config->read (String (SCIM_CONFIG_PANEL_VKB_GTK_TOOLBAR_ALWAYS_HIDDEN),
                    __config_toolbar_always_hidden);
        __config_toolbar_always_show =
            config->read (String (SCIM_CONFIG_PANEL_VKB_GTK_TOOLBAR_ALWAYS_SHOW),
                    __config_toolbar_always_show);
        __config_toolbar_auto_snap =
            config->read (String (SCIM_CONFIG_PANEL_VKB_GTK_TOOLBAR_AUTO_SNAP),
                    __config_toolbar_auto_snap);
        __config_toolbar_hide_timeout =
            config->read (String (SCIM_CONFIG_PANEL_VKB_GTK_TOOLBAR_HIDE_TIMEOUT),
                    __config_toolbar_hide_timeout);
        __config_toolbar_show_factory_icon =
            config->read (String (SCIM_CONFIG_PANEL_VKB_GTK_TOOLBAR_SHOW_FACTORY_ICON),
                    __config_toolbar_show_factory_icon);
        __config_toolbar_show_factory_name =
            config->read (String (SCIM_CONFIG_PANEL_VKB_GTK_TOOLBAR_SHOW_FACTORY_NAME),
                    __config_toolbar_show_factory_name);
        __config_toolbar_show_stick_icon =
            config->read (String (SCIM_CONFIG_PANEL_VKB_GTK_TOOLBAR_SHOW_STICK_ICON),
                    __config_toolbar_show_stick_icon);
        __config_toolbar_show_menu_icon =
            config->read (String (SCIM_CONFIG_PANEL_VKB_GTK_TOOLBAR_SHOW_MENU_ICON),
                    __config_toolbar_show_menu_icon);
        __config_toolbar_show_help_icon =
            config->read (String (SCIM_CONFIG_PANEL_VKB_GTK_TOOLBAR_SHOW_HELP_ICON),
                    __config_toolbar_show_help_icon);
        __config_toolbar_show_property_label =
            config->read (String (SCIM_CONFIG_PANEL_VKB_GTK_TOOLBAR_SHOW_PROPERTY_LABEL),
                    __config_toolbar_show_property_label);
        __config_default_sticked =
            config->read (String (SCIM_CONFIG_PANEL_VKB_GTK_DEFAULT_STICKED),
                    __config_default_sticked);
        __config_show_tray_icon =
            config->read (String (SCIM_CONFIG_PANEL_VKB_GTK_SHOW_TRAY_ICON),
                    __config_show_tray_icon);
        __config_font =
            config->read (String (SCIM_CONFIG_PANEL_VKB_GTK_FONT),
                    __config_font);
        __config_vkb_program = 
            config->read (String (SCIM_CONFIG_PANEL_VKB_GTK_VKB_PROGRAM),
                    __config_vkb_program);
        __config_vkb_parameters = 
            config->read (String (SCIM_CONFIG_PANEL_VKB_GTK_VKB_PARAMETERS),
                    __config_vkb_parameters);
        __config_vkb_window_width = 
            config->read (String (SCIM_CONFIG_PANEL_VKB_GTK_VKB_WIDTH),
                    __config_vkb_window_width);
        __config_vkb_window_height = 
            config->read (String (SCIM_CONFIG_PANEL_VKB_GTK_VKB_HEIGHT),
                    __config_vkb_window_height);
        __config_vkb_dockpanel = 
            config->read (String (SCIM_CONFIG_PANEL_VKB_GTK_DOCK_PANEL),
                    __config_vkb_dockpanel);
        __config_vkb_panel_extend = 
            config->read (String (SCIM_CONFIG_PANEL_VKB_GTK_PANEL_EXTEND),
                    __config_vkb_panel_extend);

        setup_widget_value ();

        __have_changed = false;
    }
}

    void
save_config (const ConfigPointer &config)
{
    if (!config.null ()) {
        config->write (String (SCIM_CONFIG_PANEL_VKB_GTK_TOOLBAR_ALWAYS_HIDDEN),
                __config_toolbar_always_hidden);
        config->write (String (SCIM_CONFIG_PANEL_VKB_GTK_TOOLBAR_ALWAYS_SHOW),
                __config_toolbar_always_show);
        config->write (String (SCIM_CONFIG_PANEL_VKB_GTK_TOOLBAR_AUTO_SNAP),
                __config_toolbar_auto_snap);
        config->write (String (SCIM_CONFIG_PANEL_VKB_GTK_TOOLBAR_HIDE_TIMEOUT),
                __config_toolbar_hide_timeout);
        config->write (String (SCIM_CONFIG_PANEL_VKB_GTK_TOOLBAR_SHOW_FACTORY_ICON),
                __config_toolbar_show_factory_icon);
        config->write (String (SCIM_CONFIG_PANEL_VKB_GTK_TOOLBAR_SHOW_FACTORY_NAME),
                __config_toolbar_show_factory_name);
        config->write (String (SCIM_CONFIG_PANEL_VKB_GTK_TOOLBAR_SHOW_STICK_ICON),
                __config_toolbar_show_stick_icon);
        config->write (String (SCIM_CONFIG_PANEL_VKB_GTK_TOOLBAR_SHOW_MENU_ICON),
                __config_toolbar_show_menu_icon);
        config->write (String (SCIM_CONFIG_PANEL_VKB_GTK_TOOLBAR_SHOW_HELP_ICON),
                __config_toolbar_show_help_icon);
        config->write (String (SCIM_CONFIG_PANEL_VKB_GTK_TOOLBAR_SHOW_PROPERTY_LABEL),
                __config_toolbar_show_property_label);
        config->write (String (SCIM_CONFIG_PANEL_VKB_GTK_SHOW_TRAY_ICON),
                __config_show_tray_icon);
        config->write (String (SCIM_CONFIG_PANEL_VKB_GTK_DEFAULT_STICKED),
                __config_default_sticked);
        config->write (String (SCIM_CONFIG_PANEL_VKB_GTK_FONT),
                __config_font);
        config->write (String (SCIM_CONFIG_PANEL_VKB_GTK_VKB_PROGRAM),
                __config_vkb_program);
        config->write (String (SCIM_CONFIG_PANEL_VKB_GTK_VKB_PARAMETERS),
                __config_vkb_parameters);
        config->write (String (SCIM_CONFIG_PANEL_VKB_GTK_VKB_WIDTH),
                __config_vkb_window_width);
        config->write (String (SCIM_CONFIG_PANEL_VKB_GTK_VKB_HEIGHT),
                __config_vkb_window_height);
        config->write (String (SCIM_CONFIG_PANEL_VKB_GTK_DOCK_PANEL),
                __config_vkb_dockpanel);
        config->write (String (SCIM_CONFIG_PANEL_VKB_GTK_PANEL_EXTEND),
                __config_vkb_panel_extend);

        __have_changed = false;
    }
}

    bool
query_changed ()
{
    return __have_changed;
}

    static void
on_default_spin_button_changed (GtkSpinButton *spinbutton,
        gpointer       user_data)
{
    int *value = static_cast <int *> (user_data);

    if (value) {
        *value = gtk_spin_button_get_value_as_int (spinbutton);
        __have_changed = true;
    }
}

    static void
on_default_toggle_button_toggled (GtkToggleButton *togglebutton,
        gpointer         user_data)
{
    bool *toggle = static_cast<bool*> (user_data);

    if (toggle) {
        *toggle = gtk_toggle_button_get_active (togglebutton);
        __have_changed = true;
    }
}

    static void
on_toolbar_show_behaviour_changed (GtkComboBox *combobox,
        gpointer     user_data)
{
    gint active;
    active  = gtk_combo_box_get_active (combobox);

    switch (active) {
        case SCIM_TOOLBAR_SHOW_ALWAYS:
            __config_toolbar_always_show   = true;
            __config_toolbar_always_hidden = false;
            break;
        case SCIM_TOOLBAR_SHOW_ON_DEMAND:
            __config_toolbar_always_show   = false;
            __config_toolbar_always_hidden = false;
            break;
        case SCIM_TOOLBAR_SHOW_NEVER:
            __config_toolbar_always_show   = false;
            __config_toolbar_always_hidden = true;
            break;
        default:
            __config_toolbar_always_show   = true;
            __config_toolbar_always_hidden = false;
            break;
    }

    if (__widget_toolbar_hide_timeout) {
        gtk_widget_set_sensitive (
                __widget_toolbar_hide_timeout,
                !__config_toolbar_always_hidden &&
                __config_toolbar_always_show);
    }

    if (__widget_toolbar_show_factory_icon) {
        gtk_widget_set_sensitive (
                __widget_toolbar_show_factory_icon,
                !__config_toolbar_always_hidden);
    }

    if (__widget_toolbar_show_factory_name) {
        gtk_widget_set_sensitive (
                __widget_toolbar_show_factory_name,
                !__config_toolbar_always_hidden);
    }

    if (__widget_toolbar_show_stick_icon) {
        gtk_widget_set_sensitive (
                __widget_toolbar_show_stick_icon,
                !__config_toolbar_always_hidden);
    }

    if (__widget_toolbar_show_menu_icon) {
        gtk_widget_set_sensitive (
                __widget_toolbar_show_menu_icon,
                !__config_toolbar_always_hidden);
    }

    if (__widget_toolbar_show_help_icon) {
        gtk_widget_set_sensitive (
                __widget_toolbar_show_help_icon,
                !__config_toolbar_always_hidden);
    }

    if (__widget_toolbar_show_property_label) {
        gtk_widget_set_sensitive (
                __widget_toolbar_show_property_label,
                !__config_toolbar_always_hidden);
    }

    __have_changed = true;
}

    static void
on_font_selection_clicked (GtkButton *button,
        gpointer   user_data)
{
    GtkWidget *font_selection = gtk_font_selection_dialog_new (_("Select Interface Font"));
    gint result;

    if (__config_font != _("default")) {
        gtk_font_selection_dialog_set_font_name (
                GTK_FONT_SELECTION_DIALOG (font_selection),
                __config_font.c_str ());
    }

    result = gtk_dialog_run (GTK_DIALOG (font_selection));

    if (result == GTK_RESPONSE_OK) {
        __config_font = String (
                gtk_font_selection_dialog_get_font_name (
                    GTK_FONT_SELECTION_DIALOG (font_selection)));

        gtk_button_set_label (
                GTK_BUTTON (__widget_font),
                __config_font.c_str ());

        __have_changed = true;
    }

    gtk_widget_destroy (font_selection);
}

static void
on_vkb_program_changed (GtkEntry *entry,
                            gpointer     user_data)
{
    __config_vkb_program = gtk_entry_get_text(entry);
    __have_changed = true;
}

static void
on_vkb_parameters_changed (GtkEntry *entry,
                            gpointer     user_data)
{
    __config_vkb_parameters = gtk_entry_get_text(entry);
    __have_changed = true;
}

static void
on_vkb_program_selection_clicked (GtkButton *button,
        gpointer   user_data)
{
    GtkWidget *vkb_program_selection = gtk_file_selection_new(_("Select Virtual Keyboard Program"));
    gint result;
    String tmp;
    result = gtk_dialog_run (GTK_DIALOG (vkb_program_selection));

    if (result == GTK_RESPONSE_OK) {
        struct stat filestat;
        tmp = String (
            gtk_file_selection_get_filename(GTK_FILE_SELECTION(vkb_program_selection)));
        //check if file exists and is excutable
        stat (tmp.c_str (), &filestat);
        if(S_ISREG (filestat.st_mode) &&  (S_IXUSR&filestat.st_mode))
        {
            __config_vkb_program = tmp; 
        }
        gtk_entry_set_text  (
                GTK_ENTRY (__widget_vkb_program),
                __config_vkb_program.c_str());
        __have_changed = true;
    }

    gtk_widget_destroy (vkb_program_selection);
}

static void
on_vkb_set_default(GtkButton *button,
        gpointer   user_data)
{
    __config_vkb_dockpanel             = bool (SCIM_DEFAULT_VKB_DOCKPANEL);
    __config_vkb_panel_extend          = bool (SCIM_DEFAULT_VKB_PANEL_EXTEND);
    __config_vkb_window_width          = gint (SCIM_DEFAULT_VKB_WINDOW_WIDTH);
    __config_vkb_window_height         = gint (SCIM_DEFAULT_VKB_WINDOW_HEIGHT);
    __config_vkb_program               = String (SCIM_DEFAULT_VKB_PROGRAM);
    __config_vkb_parameters            = String (SCIM_DEFAULT_VKB_PARAMETERS);
    setup_widget_value ();

    __have_changed                     = true;
}
