#include "ISHCustomSensor.h"
#include <stdlib.h>
#include <fcntl.h>
#include <unistd.h>
#include <dirent.h>
#include <errno.h>
#include <string.h>
#include <math.h>
#include "Log.h"

ISHCustomSensor::ISHCustomSensor(const struct sensor_device_t &device, const sensor_additional_information_t &information, const ish_private_data_t &data)
        :ISHSensor(device, information, data), dataSize(0)
{
        initializeDataFields();
        if (VERBOSE) {
                for (int i = 0; i < dataFieldsLength.size(); i++)
                        ALOGD("%s line: %d index: %d length: %d", __FUNCTION__, __LINE__, i, dataFieldsLength[i]);
        }
}

ISHCustomSensor::ISHCustomSensor(const Sensor * refSensor, const uint32_t serial_number)
        :ISHSensor(refSensor, serial_number), dataSize(0)
{
        initializeDataFields();
        if (VERBOSE) {
                for (int i = 0; i < dataFieldsLength.size(); i++)
                        ALOGD("%s line: %d index: %d length: %d", __FUNCTION__, __LINE__, i, dataFieldsLength[i]);
        }
}

ISHCustomSensor::~ISHCustomSensor()
{

}

bool ISHCustomSensor::initializeDataFields()
{
        DIR * dirp;
        struct dirent * entry;
        int fd, ret, error_number;

        snprintf(stringBuffer, MAX_STRING_LENGTH, SENSOR_DATA_FIELDS_DIR_FORMAT, privateData.serial_number);
        dirp = opendir(stringBuffer);
        if (dirp == NULL) {
                ALOGW("%s line: %d opendir %s error: %s", __FUNCTION__, __LINE__,
                      stringBuffer, strerror(errno));
                return false;
        }
        while ((entry = readdir(dirp)) != NULL) {
                if (entry->d_type == DT_DIR) {
                        if ((strcmp(entry->d_name, ".") == 0) || (strcmp(entry->d_name, "..") == 0))
                                continue;

                        snprintf(stringBuffer, MAX_STRING_LENGTH, SENSOR_DATA_FIELDS_DIR_FORMAT"%s/index", privateData.serial_number, entry->d_name);
                        fd = open(stringBuffer, O_RDONLY);
                        if (fd < 0) {
                                ALOGE("%s line: %d open %s error: %s", __FUNCTION__, __LINE__, stringBuffer, strerror(errno));
                                continue;
                        }
                        ret = read(fd, valueBuffer, MAX_STRING_LENGTH - 1);
                        error_number = errno;
                        close(fd);
                        if (ret < 0) {
                                ALOGE("%s line: %d read %s error: %s", __FUNCTION__, __LINE__, stringBuffer, strerror(error_number));
                                continue;
                        }
                        valueBuffer[ret] = 0;
                        int index = atoi(valueBuffer);
                        if (index < 0) {
                                ALOGE("%s line: %d invalid index: %d %s", __FUNCTION__, __LINE__, index, stringBuffer);
                                continue;
                        }

                        snprintf(stringBuffer, MAX_STRING_LENGTH, SENSOR_DATA_FIELDS_DIR_FORMAT"%s/length", privateData.serial_number, entry->d_name);
                        fd = open(stringBuffer, O_RDONLY);
                        if (fd < 0) {
                                ALOGE("%s line: %d open %s error: %s", __FUNCTION__, __LINE__, stringBuffer, strerror(errno));
                                continue;
                        }
                        ret = read(fd, valueBuffer, MAX_STRING_LENGTH - 1);
                        error_number = errno;
                        close(fd);
                        if (ret < 0) {
                                ALOGE("%s line: %d read %s error: %s", __FUNCTION__, __LINE__, stringBuffer, strerror(error_number));
                                continue;
                        }
                        valueBuffer[ret] = 0;
                        int length = atoi(valueBuffer);
                        switch (length) {
                        case 1:
                        case 2:
                        case 4:
                        case 8:
                                break;
                        default:
                                ALOGE("%s line: %d length index: %d %s", __FUNCTION__, __LINE__, length, stringBuffer);
                                continue;
                        }


                        int exponent = 1;
                        unsigned int usage_id = 0;
                        snprintf(stringBuffer, MAX_STRING_LENGTH, SENSOR_DATA_FIELDS_DIR_FORMAT"%s/exponent", privateData.serial_number, entry->d_name);
                        fd = open(stringBuffer, O_RDONLY);
                        if (fd < 0) {
                                ALOGE("%s line: %d open %s error: %s", __FUNCTION__, __LINE__, stringBuffer, strerror(errno));
                        } else {
                                ret = read(fd, valueBuffer, MAX_STRING_LENGTH - 1);
                                error_number = errno;
                                close(fd);
                                if (ret < 0) {
                                        ALOGE("%s line: %d read %s error: %s", __FUNCTION__, __LINE__, stringBuffer, strerror(error_number));
                                } else {
                                        valueBuffer[ret] = 0;
                                        exponent = atoi(valueBuffer);
                                        exponent = (exponent & 0x07) * (~(((exponent >> 4) & 0x01) - 1));
                                }
                                ALOGD_IF(DEBUG, "%s line: %d exponent: %d (%s %u)", __FUNCTION__, __LINE__, exponent, device.name, privateData.serial_number);
                        }

                        snprintf(stringBuffer, MAX_STRING_LENGTH, SENSOR_DATA_FIELDS_DIR_FORMAT"%s/usage_id", privateData.serial_number, entry->d_name);
                        fd = open(stringBuffer, O_RDONLY);
                        if (fd < 0) {
                                ALOGE("%s line: %d open %s error: %s", __FUNCTION__, __LINE__, stringBuffer, strerror(errno));
                        } else {
                                ret = read(fd, valueBuffer, MAX_STRING_LENGTH - 1);
                                error_number = errno;
                                close(fd);
                                if (ret < 0) {
                                        ALOGE("%s line: %d read %s error: %s", __FUNCTION__, __LINE__, stringBuffer, strerror(error_number));
                                } else {
                                        valueBuffer[ret] = 0;
                                        usage_id = strtoul(valueBuffer, NULL, 16);
                                }
                                ALOGD_IF(DEBUG, "%s line: %d usage_id: %x (%s %u)", __FUNCTION__, __LINE__, usage_id, device.name, privateData.serial_number);
                        }

                        if (dataFieldsLength.size() < index + 1) {
                                dataFieldsLength.resize(index + 1, length);
                                dataFieldsUsageID.resize(index + 1, length);
                                dataFieldsExponent.resize(index + 1, length);
                        } else {
                                dataFieldsLength[index] = length;
                                dataFieldsUsageID[index] = usage_id;
                                dataFieldsExponent[index] = exponent;
                        }
                        dataSize += length;
                }
        }
        closedir(dirp);

        return true;
}

size_t ISHCustomSensor::convertToEvent(const char * buf, size_t bufSize)
{
        size_t count = bufSize / dataSize;
        const char *p;

        if (bufSize % dataSize != 0) {
                ALOGE("%s line: %d Invalid size: %d unit size is: %d",
                      __FUNCTION__, __LINE__, bufSize, dataSize);
                return 0;
        }

        reallocEventsBuffer(count);

        for (int i = 0; i < count; i++) {
                p = buf + i;
                mEvents[i].timestamp = *(reinterpret_cast<const int64_t *>(p)) * US_TO_NS;
                p += sizeof(int64_t);
                p += sizeof(int64_t);
                int max_data_fields = dataFieldsLength.size() - 3;
                int j = 0, k = 0;
                float data_value;
                while (j < max_data_fields && k < 16) {
                        switch (dataFieldsLength[j + 3]) {
                        case 1:
                                data_value = *(reinterpret_cast<const char *>(p)) * pow(10, dataFieldsExponent[j + 3]);
                                p += sizeof(char);
                                break;
                        case 2:
                                data_value = *(reinterpret_cast<const short *>(p)) * pow(10, dataFieldsExponent[j + 3]);
                                p += sizeof(short);
                                break;
                        case 4:
                                data_value = *(reinterpret_cast<const int *>(p)) * pow(10, dataFieldsExponent[j + 3]);
                                p += sizeof(int);
                                break;
                        case 8:
                                data_value = *(reinterpret_cast<const int64_t *>(p)) * pow(10, dataFieldsExponent[j + 3]);
                                p += sizeof(int64_t);
                                break;
                        default:
                                ALOGE("%s line: %d invalid data field size: %d", __FUNCTION__, __LINE__, dataFieldsLength[j + 2]);
                                return 0;
                        }
                        if (dataFieldsUsageID[j + 3] != 0x00200541) {
                                mEvents[i].data[k] = data_value;
                                k++;
                        }
                        j++;
                }
        }

        return count;
}
