#include "ISHSensor.h"
#include "SessionList.h"
#include <fcntl.h>
#include <unistd.h>
#include <dirent.h>
#include <errno.h>
#include <string.h>
#include "Log.h"

const std::string props[] = {
        PROPERTY_FRIENDLY_NAME,
        PROPERTY_MINIMUM_REPORT_INTERVAL,
        PROPERTY_POWER_STATE,
        PROPERTY_REPORT_INTERVAL,
        PROPERTY_REPORT_INTERVAL_RESOLUTION,
        PROPERTY_REPORTING_STATE,
        PROPERTY_SENSOR_CONNECTION_TYPE,
        PROPERTY_SENSOR_DESCRIPTION,
        PROPERTY_SENSOR_MANUFACTURER,
        PROPERTY_SENSOR_MODEL,
        PROPERTY_SENSOR_SERIAL_NUMBER,
        PROPERTY_SENSOR_STATE,
        PROPERTY_SHARED_FIFO_COUNT,
};
const std::set<std::string> ISHSensor::propertySet(props, props + sizeof(props) / sizeof(std::string));

static SessionList &mSessionList(SessionList::getInstance());

ISHSensor::ISHSensor(const struct sensor_device_t &device, const sensor_additional_information_t &information, const ish_private_data_t &data)
        :Sensor(device, information), privateData(data), mStringBuffer(StringBuffer::getInstance())
{
        memset(&metaEvent, 0, sizeof(metaEvent));
        metaEvent.version = META_DATA_VERSION;
        metaEvent.type = SENSOR_TYPE_META_DATA;
        metaEvent.meta_data.what = META_DATA_FLUSH_COMPLETE;
        metaEvent.meta_data.sensor = device.handle;

        if ((privateData.sensitivity == SENSITIVITY_DEFAULT) && ((device.flags & ~SENSOR_FLAG_WAKE_UP & REPORTING_MODE_MASK) == SENSOR_FLAG_CONTINUOUS_MODE))
                privateData.sensitivity = 0;

        std::string shared_fifo_count;
        if (getHIDCommonProperty(PROPERTY_SHARED_FIFO_COUNT, shared_fifo_count)) {
                if ((device.flags & ~SENSOR_FLAG_WAKE_UP & REPORTING_MODE_MASK) == SENSOR_FLAG_ONE_SHOT_MODE)
                        this->device.fifoMaxEventCount = 0;
                else
                        sscanf(shared_fifo_count.c_str(), "%u", &this->device.fifoMaxEventCount);
                ALOGD_IF(DEBUG, "(%s %d) fifoMaxEventCount: %d", __FUNCTION__, __LINE__, this->device.fifoMaxEventCount);
        }

        ALOGD_IF(DEBUG, "(%s %d) serial number: %X friendly_name: %s", __FUNCTION__, __LINE__, data.serial_number, data.friendly_name.c_str());
        std::string minimum_interval;
        if (getHIDCommonProperty(PROPERTY_MINIMUM_REPORT_INTERVAL, minimum_interval)) {
                sscanf(minimum_interval.c_str(), "%u", &privateData.minimum_report_interval);
        } else {
                privateData.minimum_report_interval = 0;
        }

        getHIDCommonProperty(PROPERTY_FRIENDLY_NAME, privateData.friendly_name);

        ALOGD_IF(DEBUG, "(%s %d) minimum_interval: %u friendly_name: %s", __FUNCTION__, __LINE__, privateData.minimum_report_interval, privateData.friendly_name.c_str());
        if ((this->device.minDelay > 0) && (device.minDelay < privateData.minimum_report_interval * 1000)) {
                ALOGD_IF(DEBUG, "(%s %d) configured minDelay: %d changed minDelay: %d", __FUNCTION__, __LINE__, device.minDelay, privateData.minimum_report_interval * 1000);
                this->device.minDelay = privateData.minimum_report_interval * 1000;
        }

        std::string vendor;
        if (getHIDCommonProperty(PROPERTY_SENSOR_MANUFACTURER, vendor)) {
                ALOGD_IF(DEBUG, "(%s %d) vendor: %s", __FUNCTION__, __LINE__, vendor.c_str());
                this->device.vendor = mStringBuffer.addString(vendor.c_str());
        }
}

ISHSensor::ISHSensor(const Sensor * refSensor, const uint32_t serial_number)
        :Sensor(refSensor), mStringBuffer(StringBuffer::getInstance())
{
        privateData = (reinterpret_cast<const ISHSensor *>(refSensor))->privateData;
        privateData.serial_number = serial_number;
}

bool ISHSensor::checkHIDCommonPropertyValid(const char *propertyName)
{
        if (!propertySet.count(propertyName)) {
                ALOGE("%s line: %d Cannot find the property: %s", __FUNCTION__, __LINE__, propertyName);
                return false;
        }

        return true;
}


bool ISHSensor::setHIDCommonProperty(const char *propertyName, const char *propertyValue, int propertyLength)
{
        snprintf(stringBuffer, MAX_STRING_LENGTH, SENSOR_PROP_VALUE_FORMAT, privateData.serial_number, propertyName);
        stringBuffer[MAX_STRING_LENGTH - 1] = 0;
        int fd = open(stringBuffer, O_WRONLY);
        if (fd < 0) {
                ALOGE("%s line: %d Open %s error: %s", __FUNCTION__, __LINE__, stringBuffer, strerror(errno));
                return false;
        }

        int ret = write(fd, propertyValue, propertyLength);
        if (ret < 0) {
                ALOGE("%s line: %d Write to %s error: %s", __FUNCTION__, __LINE__, stringBuffer, strerror(errno));
                close(fd);
                return false;
        }

        close(fd);

        return true;
}

bool ISHSensor::getHIDCommonProperty(const char *propertyName, std::string &propertyValue)
{
        snprintf(stringBuffer, MAX_STRING_LENGTH, SENSOR_PROP_VALUE_FORMAT, privateData.serial_number, propertyName);
        stringBuffer[MAX_STRING_LENGTH - 1] = 0;
        int fd = open(stringBuffer, O_RDONLY);
        if (fd < 0) {
                ALOGE("%s line: %d Open %s error: %s", __FUNCTION__, __LINE__, stringBuffer, strerror(errno));
                return false;
        }

        int ret = read(fd, valueBuffer, MAX_STRING_LENGTH - 1);
        if (ret < 0) {
                ALOGE("%s line: %d read from %s error: %s", __FUNCTION__, __LINE__, stringBuffer, strerror(errno));
                close(fd);
                return false;
        }
        valueBuffer[ret] = 0;
        propertyValue = valueBuffer;
        close(fd);

        return true;
}

bool ISHSensor::startStreaming(Session * session, int64_t samplingPeriodUs, int64_t maxReportLatencyUs)
{
        bool result, params_changed = false;

        if ((samplingPeriodUs != bestBatchParams.batchDelay) || (maxReportLatencyUs != bestBatchParams.batchTimeout)) {
                params_changed = true;
        }

        result = Sensor::startStreaming(session, samplingPeriodUs, maxReportLatencyUs);
        if (!result)
                return result;

        if (params_changed) {
                ALOGD_IF(DEBUG, "%s line: %d sensor: %s samplingPeriodUs: %lld maxReportLatencyUs: %lld sensitivity: %d",
                         __FUNCTION__, __LINE__, device.name, samplingPeriodUs, maxReportLatencyUs, privateData.sensitivity);
                if (privateData.sensitivity != SENSITIVITY_DEFAULT)
                        result = setSensitivity(privateData.sensitivity);
                result = (result && startStopImpl(true));
        }

        return result;
}

bool ISHSensor::stopStreaming(Session * session)
{
        bool result = Sensor::stopStreaming(session);

        if (!result)
                return result;

        if ((bestBatchParams.batchDelay != -1) && (bestBatchParams.batchTimeout != -1)) {
                ALOGD_IF(DEBUG, "%s line: %d sensor: %s samplingPeriodUs: %lld maxReportLatencyUs: %lld",
                         __FUNCTION__, __LINE__, device.name, bestBatchParams.batchDelay, bestBatchParams.batchTimeout);
                result = startStopImpl(true);
        } else {
                ALOGD_IF(DEBUG, "%s line: %d sensor: %s", __FUNCTION__, __LINE__, device.name);
                result = startStopImpl(false);
        }

        return result;
}

bool ISHSensor::restartStreaming() {
        bool result = true;

        if (batchParams.size() > 0) {
                if (privateData.sensitivity != SENSITIVITY_DEFAULT)
                        result = setSensitivity(privateData.sensitivity);
                result = (result && startStopImpl(true));
                if (!result)
                        ALOGE("%s line: %d sensor: %s restart failed!", __FUNCTION__, __LINE__, device.name);

                result = setPrivateProperty();
                if (!result)
                        ALOGE("%s line: %d sensor: %s setPrivateProperty failed!", __FUNCTION__, __LINE__, device.name);
        }

        return result;
}

bool ISHSensor::startStopImpl(bool start)
{
        bool result = true;

        if (start) {
                // Active sensor
                snprintf(valueBuffer, MAX_STRING_LENGTH, "%u", USAGE_SENSOR_PROPERTY_POWER_STATE_D0_FULL_POWER_ENUM);
                valueBuffer[MAX_STRING_LENGTH - 1] = 0;
                result = setHIDCommonProperty(PROPERTY_POWER_STATE, valueBuffer, strlen(valueBuffer));

                // Set delay and timeout
                int delay_ms = bestBatchParams.batchDelay / 1000; // transform microsecond to millisecond
                snprintf(valueBuffer, MAX_STRING_LENGTH, "%d", delay_ms);
                valueBuffer[MAX_STRING_LENGTH - 1] = 0;
                result = (result && setHIDCommonProperty(PROPERTY_REPORT_INTERVAL, valueBuffer, strlen(valueBuffer)));

                int timeout_ms = bestBatchParams.batchTimeout / 1000; // transform microsecond to millisecond
                snprintf(valueBuffer, MAX_STRING_LENGTH, "%d", timeout_ms);
                valueBuffer[MAX_STRING_LENGTH - 1] = 0;
                result = (result && setHIDCommonProperty(PROPERTY_REPORT_INTERVAL_RESOLUTION, valueBuffer, strlen(valueBuffer)));

                if (device.flags & SENSOR_FLAG_WAKE_UP)
                        snprintf(valueBuffer, MAX_STRING_LENGTH, "%u", USAGE_SENSOR_PROPERTY_REPORTING_STATE_ALL_EVENTS_WAKE_ENUM);
                else
                        snprintf(valueBuffer, MAX_STRING_LENGTH, "%u", USAGE_SENSOR_PROPERTY_REPORTING_STATE_ALL_EVENTS_ENUM);
                valueBuffer[MAX_STRING_LENGTH - 1] = 0;
                result = (result && setHIDCommonProperty(PROPERTY_REPORTING_STATE, valueBuffer, strlen(valueBuffer)));

        } else {
                snprintf(valueBuffer, MAX_STRING_LENGTH, "%u", USAGE_SENSOR_PROPERTY_REPORTING_STATE_NO_EVENTS_ENUM);
                valueBuffer[MAX_STRING_LENGTH - 1] = 0;
                result = setHIDCommonProperty(PROPERTY_REPORTING_STATE, valueBuffer, strlen(valueBuffer));

                // Reset delay
                result = (result && setHIDCommonProperty(PROPERTY_REPORT_INTERVAL, "0", strlen("0")));
                result = (result && setHIDCommonProperty(PROPERTY_REPORT_INTERVAL_RESOLUTION, "0", strlen("0")));

                // Deactive sensor
                snprintf(valueBuffer, MAX_STRING_LENGTH, "%u", USAGE_SENSOR_PROPERTY_POWER_STATE_D1_LOW_POWER_ENUM);
                valueBuffer[MAX_STRING_LENGTH - 1] = 0;
                result = (result && setHIDCommonProperty(PROPERTY_POWER_STATE, valueBuffer, strlen(valueBuffer)));
        }

        return result;
}

bool ISHSensor::flushStreaming(Session * session)
{
        snprintf(stringBuffer, MAX_STRING_LENGTH, SENSOR_FLUSH_NODE_FORMAT, privateData.serial_number);
        stringBuffer[MAX_STRING_LENGTH - 1] = 0;
        int fd = open(stringBuffer, O_RDONLY);
        if (fd < 0) {
                ALOGE("%s line: %d Open %s error: %s", __FUNCTION__, __LINE__, stringBuffer, strerror(errno));
                return false;
        }

        int ret = read(fd, stringBuffer, MAX_STRING_LENGTH);
        if (ret < 0) {
                ALOGE("%s line: %d Read sysfs error: %s", __FUNCTION__, __LINE__, strerror(errno));
                close(fd);
                return false;
        }

        close(fd);

        return true;
}

void ISHSensor::removeSession(Session *session)
{
        Sensor::removeSession(session);
        privatePropertiesTable.erase(session);
        arbitratePrivateProperty(NULL, NULL);
}

bool ISHSensor::setProperty(Session * session, void * property, size_t size)
{
        if (size < sizeof(unsigned int)) {
                ALOGE("%s line: %d Invalid property size: %d", __FUNCTION__, __LINE__, size);
                return false;
        }

        if (!session->sensorIsActivated(device.handle))
                session->activateSensor(device.handle, this, true);

        unsigned int *property_type = reinterpret_cast<unsigned int *>(property);
        struct ish_private_property *private_property = NULL;
        switch(*property_type) {
        case ISH_PRIVATE_PROPERTY:
                if (size < sizeof(unsigned int) + sizeof(struct ish_private_property)) {
                        ALOGE("%s line: %d Invalid property size: %d", __FUNCTION__, __LINE__, size);
                        return false;
                }
                private_property = reinterpret_cast<struct ish_private_property *>(reinterpret_cast<char *>(property) + sizeof(unsigned int));
                arbitratePrivateProperty(session, private_property);
                return setPrivateProperty();
        case ISH_SMHI_PROPERTY:
                break;
        default:
                ALOGE("%s line: %d Invalid property type: %u", __FUNCTION__, __LINE__, *property_type);
                return false;
        }

        return false;
}

size_t ISHSensor::getProperty(Session * session, void * command, size_t commandSize, void **property)
{
        if (commandSize < sizeof(unsigned int)) {
                ALOGE("%s line: %d Invalid property command size: %d", __FUNCTION__, __LINE__, commandSize);
                return false;
        }

        if (!session->sensorIsActivated(device.handle))
                session->activateSensor(device.handle, this, true);

        unsigned int *property_type = reinterpret_cast<unsigned int *>(command);
        unsigned int *usage_id = NULL;
        switch(*property_type) {
        case ISH_PRIVATE_PROPERTY:
                if (commandSize < sizeof(unsigned int) + sizeof(unsigned int)) {
                        ALOGE("%s line: %d Invalid property size: %d", __FUNCTION__, __LINE__, commandSize);
                        return 0;
                }
                usage_id = reinterpret_cast<unsigned int *>(reinterpret_cast<char *>(command) + sizeof(unsigned int));
                return getPrivateProperty(*usage_id, property);
        case ISH_SMHI_PROPERTY:
                break;
        default:
                ALOGE("%s line: %d Invalid property type: %u", __FUNCTION__, __LINE__, *property_type);
                return 0;
        }

        return 0;
}

size_t ISHSensor::convertToStreaming(const char * buf, size_t bufSize)
{
        if (mStreamingSize < bufSize) {
                if (mStreaming != NULL)
                        delete [] mStreaming;
                mStreaming = new char[bufSize];
                mStreamingSize = bufSize;
        }
        memcpy(mStreaming, buf, bufSize);
        return bufSize;
}

void ISHSensor::handleEvents(int fd)
{

}

void ISHSensor::handleEvents(const char * buf, size_t bufSize)
{
        const struct sensor_colloection_data_t *data = reinterpret_cast<const struct sensor_colloection_data_t *>(buf);
        size_t eventsCount = 0;
        size_t streamingCount = 0;
        std::map<Session *, BatchParams>::iterator it;
        unsigned int flag = 0;

        if (data->id & PSEUSDO_EVENT_BIT) {
                flag = FLUSH_COMPLETE_BIT & *(reinterpret_cast<const unsigned int *>(data->buf));
        }

        mSessionList.lockSessions();
        lockbatchParams(); // lock batchParams for it may be changed in main thread
        for (it = batchParams.begin(); it != batchParams.end(); it++) {
                Session *session = it->first;
                if (session->getCategory() == ANDROID_STANDARD) {
                        if (flag) {
                                session->dispatchEvents(sizeof(sensors_meta_data_event_t), &metaEvent);
                        } else if (eventsCount == 0) {
                                eventsCount = convertToEvent(data->buf, data->size - sizeof(struct sensor_colloection_data_t));
                        }
                        if (eventsCount != 0)
                                session->dispatchEvents(eventsCount * sizeof(struct sensors_event_t), mEvents);
                } else if (session->getCategory() == AWARE_SERVICE) {
                        if ((!flag) && (streamingCount == 0))
                                streamingCount = convertToStreaming(data->buf, data->size - sizeof(struct sensor_colloection_data_t));
                        session->dispatchEvents(streamingCount, mStreaming);
                }
        }
        unlockbatchParams();
        mSessionList.unlockSessions();
}

bool  ISHSensor::setSensitivity(const char *name, const unsigned int value)
{
        if (name != NULL) {
                if (strstr(name,"sensitivity") == NULL)
                        ALOGW("%s line: %d %s may not a sensitivity!", __FUNCTION__, __LINE__, name);
                snprintf(stringBuffer, MAX_STRING_LENGTH, SENSOR_PROP_VALUE_FORMAT, privateData.serial_number, name);
                stringBuffer[MAX_STRING_LENGTH - 1] = 0;
        } else {
                char *sensitivity_name = NULL;

                snprintf(stringBuffer, MAX_STRING_LENGTH, SENSOR_PROP_DIR_FORMAT, privateData.serial_number);
                stringBuffer[MAX_STRING_LENGTH - 1] = 0;

                DIR *dirp = opendir(stringBuffer);
                if (dirp == NULL) {
                        ALOGE("%s line: %d can't open directory %s!", __FUNCTION__, __LINE__, stringBuffer);
                        return false;
                }

                struct dirent * entry;
                while ((entry = readdir(dirp)) != NULL) {
                        if (strstr(entry->d_name, PROPERTY_SENSITIVITY_SUFFIX) != NULL) {
                                sensitivity_name = entry->d_name;
                                break;
                        }
                }
                if (sensitivity_name != NULL) {
                        snprintf(stringBuffer, MAX_STRING_LENGTH, SENSOR_PROP_VALUE_FORMAT, privateData.serial_number, sensitivity_name);
                        stringBuffer[MAX_STRING_LENGTH - 1] = 0;
                }
                closedir(dirp);
        }

        int fd = open(stringBuffer, O_WRONLY);
        if (fd < 0) {
                ALOGE("%s line: %d Open %s error: %s", __FUNCTION__, __LINE__, stringBuffer, strerror(errno));
                return false;
        }

        snprintf(valueBuffer, MAX_STRING_LENGTH, "%u", value);
        valueBuffer[MAX_STRING_LENGTH - 1] = 0;
        int ret = write(fd, valueBuffer, strlen(valueBuffer));
        if (ret < 0) {
                ALOGE("%s line: %d Write sysfs error: %s", __FUNCTION__, __LINE__, strerror(errno));
                close(fd);
                return false;
        }

        close(fd);

        return true;
}

bool ISHSensor::setSensitivity(const unsigned int value)
{
        return setSensitivity(NULL, value);
}

bool ISHSensor::arbitratePrivateProperty(Session *session, struct ish_private_property *private_property)
{
        if ((session != NULL) && (private_property != NULL)) {
                privatePropertiesTable.insert(std::pair<Session *, struct ish_private_property>(session, *private_property));
                currentPrivateProperties.clear();
                currentPrivateProperties.push_back(*private_property);
        }

        return true;
}

bool ISHSensor::setPrivateProperty()
{
        bool result = true;

        std::list<struct ish_private_property>::iterator it;
        for (it = currentPrivateProperties.begin(); it != currentPrivateProperties.end(); it++) {
                unsigned int usage_id = it->usage_id;
                int property_value = it->property_value;
                if (privateData.private_property_usage_ids.count(usage_id) == 0) {
                        ALOGE("%s line: %d sensor: %s can't find usage id: %u", __FUNCTION__, __LINE__, device.name, usage_id);
                        result = false;
                        continue;
                }

                snprintf(stringBuffer, MAX_STRING_LENGTH, SENSOR_PRIVATE_PROP_VALUE_FORMAT, privateData.serial_number, usage_id);
                stringBuffer[MAX_STRING_LENGTH - 1] = 0;
                int fd = open(stringBuffer, O_WRONLY);
                if (fd < 0) {
                        ALOGE("%s line: %d Open %s error: %s", __FUNCTION__, __LINE__, stringBuffer, strerror(errno));
                        result = false;
                        continue;
                }

                snprintf(valueBuffer, MAX_STRING_LENGTH, "%d", property_value);
                valueBuffer[MAX_STRING_LENGTH - 1] = 0;
                int ret = write(fd, valueBuffer, strlen(valueBuffer) + 1);
                if (ret < 0) {
                        ALOGE("%s line: %d Write to %s error: %s", __FUNCTION__, __LINE__, stringBuffer, strerror(errno));
                        close(fd);
                        result = false;
                        continue;
                }

                close(fd);
        }

        return result;
}

size_t ISHSensor::getPrivateProperty(unsigned int usage_id, void **property)
{
        if (privateData.private_property_usage_ids.count(usage_id) == 0) {
                ALOGE("%s line: %d sensor: %s can't find usage id: %u", __FUNCTION__, __LINE__, device.name, usage_id);
                return 0;
        }

        snprintf(stringBuffer, MAX_STRING_LENGTH, SENSOR_PRIVATE_PROP_VALUE_FORMAT, privateData.serial_number, usage_id);
        stringBuffer[MAX_STRING_LENGTH - 1] = 0;
        int fd = open(stringBuffer, O_RDONLY);
        if (fd < 0) {
                ALOGE("%s line: %d Open %s error: %s", __FUNCTION__, __LINE__, stringBuffer, strerror(errno));
                return 0;
        }

        int ret = read(fd, valueBuffer, MAX_STRING_LENGTH);
        close(fd);
        if (ret < 0) {
                ALOGE("%s line: %d Read from %s error: %s", __FUNCTION__, __LINE__, stringBuffer, strerror(errno));
                return 0;
        }
        if (ret == 0) {
                ALOGE("%s line: %d Read from %s error: return size is 0", __FUNCTION__, __LINE__, stringBuffer);
                return 0;
        }

        *property = new char[ret];
        memcpy(*property, valueBuffer, ret);

        return 0;
}

bool ISHSensor::setSMHIProperty(unsigned char *buf, int len)
{
        sc_sdt_info_entry *entry = reinterpret_cast<sc_sdt_info_entry *>(malloc(sizeof(sc_sdt_info_entry) + len));
        if (entry == NULL) {
                ALOGE("%s line: %d malloc error!", __FUNCTION__, __LINE__);
                return false;
        }
        entry->info_id = SC_SDT_CALI_INFO;
        entry->reserved = 0;
        entry->data_length = len;
        memcpy(entry->sdt_info_data, buf, len);

        int handle = smhi_client_connect();
        if (handle < 0) {
                ALOGE("%s line: %d smhi_client_connect error!", __FUNCTION__, __LINE__);
                free(entry);
                return false;
        }

        if (smhi_client_set_sensor_conf(handle, privateData.luid, ISH_CONFIG_CALIBRATION, 1, entry, len) < 0) {
                ALOGE("%s line: %d smhi_client_set_sensor_conf error!", __FUNCTION__, __LINE__);
                smhi_client_disconnect(handle);
                free(entry);
                return false;
        }

        smhi_client_disconnect(handle);
        return true;
}
