#ifndef _SC_SENSOR_LUID_H_
#define _SC_SENSOR_LUID_H_

#include <sys/types.h>
#include <stdint.h>

#ifdef __cplusplus
extern "C" {
#endif

typedef uint16_t sc_sensor_type; /**< Represents type of sensor. SC needs to know what type of sensor is being connected to the system. See sc_sensor_luid. Refer to SC_SENSOR_TYPE_xxx macro values for different sensor types.*/
typedef uint16_t sc_sensor_vendor_id; /**< Vendor ID for the sensor. SC needs to know the sensor vendor for each sensor while registering the sensor and also while passing commands and requests. Refer to sc_sensor_luid. Refer to SC_SENSOR_VENDOR_xxx macro values for different sensor vendors*/
typedef uint16_t sc_sensor_sub_type; /**< Sensor Sub Type (Product ID) for the sensor. SC distinguishes different sensor parts of the same type belonging to the same sensor vendor using the sensor sub type field. See sc_sensor_luid. */
typedef uint8_t  sc_instance_id; /**< Instance ID of sensor. Every sensor type can find multiple usages on the same platform. Example: two accelerometer sensors present on a platform; having one instance on the display panel of the notebook and another at the base. Both are exposed as two different instances of the accelerometer sensor. */

/**
*****************************************************************************
* @ingroup sensor_core_api
* Sensor's LUID information.
*
* LUID is short form of Locally Unique ID. This is a unique ID, which ISH will
* use to map different types of sensors in the system and differentiate amongst them.
*
*****************************************************************************/
typedef struct _sc_sensor_luid
{
        sc_sensor_type      sensor_type; /**< Type of Sensor, i.e. accelerometer or barometer etc. */
        sc_sensor_vendor_id vendor_id; /**< Vendor ID. Defines the ID of the vendor who has created this sensor */
        sc_sensor_sub_type  sensor_sub_type; /**< Product ID. Differentiate products from the same vendor */
        sc_instance_id      instance_id; /**< Instance of sensor */
        uint8_t             luid_flags; /**< Custom flags */
} sc_sensor_luid;
/** @addtogroup sensor_core_api
@{
*/
/*LUID flag definitions*/

#define SC_LUID_FLAG_NONE        0 /**< LUID flag NULL */
#define SC_LUID_FLAG_CALIBRATED  (1 << 0) /**< LUID flag indicates Sensor is calibrated */
#define SC_LUID_FLAG_PHYSICAL    (1 << 1) /**< LUID flag indicates Sensor is not calibrated */
#define SC_LUID_FLAG_MS_EXT      (1 << 2) /**< LUID flag indicates Sensor is exposed externally and follows Microsoft definition */
#define SC_LUID_FLAG_ANDROID_EXT (1 << 3) /**< LUID flag indicates Sensor is exposed externally and follows Android definition */
#define SC_LUID_FLAG_SYNTHETIC   (1 << 4)

// internal flags - do not specifically identify a unique sensor
#define SC_LUID_FLAG_NOT_EXPOSED (1 << 6) /**< LUID flag indicates Sensor shouldn't be exposed to host */
#define SC_LUID_FLAG_CUSTOM      (1 << 7) /**< LUID flag indicates Sensor is exposed externally as custom sensor */

#define SC_LUID_FLAGS_INTERNALS (SC_LUID_FLAG_NOT_EXPOSED | SC_LUID_FLAG_CUSTOM)
/** @} */

#define LUID_STRING_LENGTH 16
#define LUID_STRING_PREFIX_LENGTH 5
sc_sensor_luid luid_from_string(const char * str);
char * luid_to_string(sc_sensor_luid luid, char * str);

#ifdef __cplusplus
}
#endif

#endif
