/*
 * Copyright (c) 2015 Intel Corporation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "PSHService.h"
#include "ISHService.h"
#include "IIOService.h"
#include "SensorList.h"
#include "SessionList.h"
#include "SensorEnumerator.h"
#include "sensorhubd.h"
#include "libsensorhub.h"
#include "message.h"
#include <unistd.h>
#include <pthread.h>
#include <errno.h>
#include <string.h>
#include "Log.h"
#include <sys/socket.h>
#include <sys/un.h>
//#include <cutils/sockets.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <signal.h>
#include <map>
#include <queue>

static SensorList &mSensorList(SensorList::getInstance());
static SessionList &mSessionList(SessionList::getInstance());
const std::vector<PlatformService *> &serviceList = mSensorList.getServiceList();

static void * sensorEventsListener(void * arg)
{
        std::map<int, PlatformService *> platformServiceTable;
        std::queue<int> dataFdQueue;
        fd_set readFds, exceptionFds, rFds, eFds;
        int maxFd = 0;
        int ret;
        dataSelectMode selectMode = SELECT_MODE_READ;
        FD_ZERO(&readFds);
        FD_ZERO(&exceptionFds);

        for (int i = 0 ; i < serviceList.size(); i++) {
                PlatformService *service = serviceList[i];
                service->getDataFds(dataFdQueue);
                selectMode = service->getDataSelectMode();
                while (!dataFdQueue.empty()) {
                        int fd = dataFdQueue.front();
                        dataFdQueue.pop();
                        if (fd < 0) {
                                ALOGE("%s line: %d invalid fd: %d", __FUNCTION__, __LINE__, fd);
                                continue;
                        }
                        platformServiceTable[fd] = service;
                        if (selectMode == SELECT_MODE_EXCEPTION)
                                FD_SET(fd, &exceptionFds);
                        else
                                FD_SET(fd, &readFds);

                        if (fd > maxFd)
                                maxFd = fd;
                }
        }
        if (platformServiceTable.size() == 0) {
                ALOGE("%s line: %d no valid platform service!", __FUNCTION__, __LINE__);
                return NULL;
        }
        ALOGE("%s line: %d", __FUNCTION__, __LINE__);
        while (true) {
                rFds = readFds;
                eFds = exceptionFds;
                int ret = select(maxFd + 1, &readFds, NULL, &exceptionFds, NULL);
                if (ret < 0) {
                        if (errno != EINTR)
                                ALOGE("%s line: %d select error: %s", __FUNCTION__, __LINE__, strerror(errno));
                        continue;
                }

                for (int fd = maxFd; fd >= 0; fd--) {
                        if (FD_ISSET(fd, &rFds) || FD_ISSET(fd, &eFds))
                                platformServiceTable[fd]->handleEvents(fd);
                }
        }

        return NULL;
}

#define COMMON_ACK_DATA_MAX_SIZE 32
typedef struct {
        sensorhub_message_header_t header;
        uint8_t data[COMMON_ACK_DATA_MAX_SIZE];
} __attribute__ ((packed)) common_ack_buf_t;

static void sendMessageAcknowledge(int fd, int acknowledgeNumber, int ackDataSize, uint8_t * ackData)
{
        static common_ack_buf_t commonAckBuf;
        uint8_t * send_buf = NULL;
        sensorhub_message_header_t * header;

        if (ackDataSize > COMMON_ACK_DATA_MAX_SIZE) {
                send_buf = new uint8_t[sizeof(sensorhub_message_header_t) + ackDataSize];
        } else {
                send_buf = reinterpret_cast<uint8_t *>(&commonAckBuf);
        }
        header = reinterpret_cast<sensorhub_message_header_t *>(send_buf);
        header->require_code = REQUIRE_NONE;
        header->ack = 1;
        header->acknowledge_number = acknowledgeNumber;
        header->data_size = ackDataSize;
        if (ackData != NULL)
                memcpy(header->data, ackData, ackDataSize);

        if (send(fd, reinterpret_cast<void *>(send_buf), sizeof(sensorhub_message_header_t) + ackDataSize, 0) < 0) {
                ALOGE("%s line: %d: send message to client error: %s", __FUNCTION__, __LINE__, strerror(errno));
        }

        if (ackDataSize > COMMON_ACK_DATA_MAX_SIZE)
                delete [] send_buf;
}

static void handleCreateSessionDataFd(int fd)
{
        Session * session = mSessionList.createSession(fd);
        int ackDataSize = sizeof(session);
        uint8_t * ackData = new uint8_t[ackDataSize];
        memcpy(ackData, &session, ackDataSize);

        sendMessageAcknowledge(fd, 0, ackDataSize, ackData);
        if (ackData != NULL)
                delete [] ackData;
}

static void handleCreateSessionCtrlFd(int fd, int dataSize, uint8_t * data)
{
        Session * session;
        int ackDataSize = 1;
        uint8_t ackData = 0;
        session_category_t category;

        if (data == NULL) {
                ALOGE("%s line: %d session is NULL!", __FUNCTION__, __LINE__);
        } else if (dataSize < sizeof(session) + sizeof(session_category_t)) {
                ALOGE("%s line: %d invalid session size!", __FUNCTION__, __LINE__);
        } else {
                memcpy(&session, data, sizeof(session));
                category = *(reinterpret_cast<session_category_t *>(data + sizeof(session)));
                if (mSessionList.sessionExists(session)) {
                        session->setControlFd(fd);
                        session->setCategory(category);
                        mSessionList.updateSession(session);
                        ackData = 1;
                } else {
                        ALOGE("%s line: %d cannot find session: %p!", __FUNCTION__, __LINE__, session);
                }
        }
        sendMessageAcknowledge(fd, dataSize, ackDataSize, &ackData);
}

static void handleStartStreaming(int fd, int dataSize, uint8_t * data)
{
        Session * session;
        Sensor * sensor;
        start_streaming_data_t * startStreamingData;
        int ackDataSize = 1;
        uint8_t ackData = 0;

        if (data == NULL) {
                ALOGE("%s line: %d data is NULL!", __FUNCTION__, __LINE__);
        } else if (dataSize < sizeof(start_streaming_data_t)) {
                ALOGE("%s line: %d invalid data size!", __FUNCTION__, __LINE__);
        } else {
                startStreamingData = reinterpret_cast<start_streaming_data_t *>(data);

                session = mSessionList.getSession(fd);
                if (session == NULL) {
                        ALOGE("%s line: %d session has not been created!", __FUNCTION__, __LINE__);
                        goto error_start_streaming;
                }

                sensor = mSensorList.getSensor(startStreamingData->handle);
                if (sensor == NULL) {
                        ALOGE("%s line: %d sensor: %d does not exists!", __FUNCTION__, __LINE__, startStreamingData->handle);
                        goto error_start_streaming;
                }

                bool result = sensor->startStreaming(session, startStreamingData->sampling_period_us, startStreamingData->max_report_latency_us);
                if (result) {
                        ackData = 1;
                } else {
                        ALOGE("%s line: %d sensor: %d start streaming failed!", __FUNCTION__, __LINE__, startStreamingData->handle);
                }
        }

error_start_streaming:
        sendMessageAcknowledge(fd, dataSize, ackDataSize, &ackData);
}

static void handleStopStreaming(int fd, int dataSize, uint8_t * data)
{
        Session * session;
        Sensor * sensor;
        int ackDataSize = 1;
        uint8_t ackData = 0;

        if (data == NULL) {
                ALOGE("%s line: %d data is NULL!", __FUNCTION__, __LINE__);
        } else if (dataSize < sizeof(int)) {
                ALOGE("%s line: %d invalid data size!", __FUNCTION__, __LINE__);
        } else {
                int handle = *(reinterpret_cast<int *>(data));

                session = mSessionList.getSession(fd);
                if (session == NULL) {
                        ALOGE("%s line: %d session has not been created!", __FUNCTION__, __LINE__);
                        goto error_stop_streaming;
                }

                sensor = mSensorList.getSensor(handle);
                if (sensor == NULL) {
                        ALOGE("%s line: %d sensor: %d does not exists!", __FUNCTION__, __LINE__, handle);
                        goto error_stop_streaming;
                }

                bool result = sensor->stopStreaming(session);
                if (result) {
                        ackData = 1;
                } else {
                        ALOGE("%s line: %d sensor: %d stop streaming failed!", __FUNCTION__, __LINE__, handle);
                }
        }

error_stop_streaming:
        sendMessageAcknowledge(fd, dataSize, ackDataSize, &ackData);
}

static void handleFlushStreaming(int fd, int dataSize, uint8_t * data)
{
        Session * session;
        Sensor * sensor;
        int ackDataSize = 1;
        uint8_t ackData = 0;

        if (data == NULL) {
                ALOGE("%s line: %d data is NULL!", __FUNCTION__, __LINE__);
        } else if (dataSize < sizeof(int)) {
                ALOGE("%s line: %d invalid data size!", __FUNCTION__, __LINE__);
        } else {
                int handle = *(reinterpret_cast<int *>(data));

                session = mSessionList.getSession(fd);
                if (session == NULL) {
                        ALOGE("%s line: %d session has not been created!", __FUNCTION__, __LINE__);
                        goto error_flush_streaming;
                }

                sensor = mSensorList.getSensor(handle);
                if (sensor == NULL) {
                        ALOGE("%s line: %d sensor: %d does not exists!", __FUNCTION__, __LINE__, handle);
                        goto error_flush_streaming;
                }

                bool result = sensor->flushStreaming(session);
                if (result) {
                        ackData = 1;
                } else {
                        ALOGE("%s line: %d sensor: %d flush streaming failed!", __FUNCTION__, __LINE__, handle);
                }
        }

error_flush_streaming:
        sendMessageAcknowledge(fd, dataSize, ackDataSize, &ackData);
}

static void handleSetProperty(int fd, int dataSize, uint8_t * data)
{
        Session * session;
        Sensor * sensor;
        int ackDataSize = 1;
        uint8_t ackData = 0;

        if (data == NULL) {
                ALOGE("%s line: %d data is NULL!", __FUNCTION__, __LINE__);
        } else if (dataSize < sizeof(int)) {
                ALOGE("%s line: %d invalid data size!", __FUNCTION__, __LINE__);
        } else {
                int handle = *(reinterpret_cast<int *>(data));

                session = mSessionList.getSession(fd);
                if (session == NULL) {
                        ALOGE("%s line: %d session has not been created!", __FUNCTION__, __LINE__);
                        goto error_set_property;
                }

                sensor = mSensorList.getSensor(handle);
                if (sensor == NULL) {
                        ALOGE("%s line: %d sensor: %d does not exists!", __FUNCTION__, __LINE__, handle);
                        goto error_set_property;
                }

                bool result = sensor->setProperty(session, data + sizeof(int), dataSize - sizeof(int));
                if (result) {
                        ackData = 1;
                } else {
                        ALOGE("%s line: %d sensor: %d set property failed!", __FUNCTION__, __LINE__, handle);
                }
        }

error_set_property:
        sendMessageAcknowledge(fd, dataSize, ackDataSize, &ackData);
}

static void handleGetProperty(int fd, int dataSize, uint8_t * data)
{
        Session * session;
        Sensor * sensor;
        uint8_t * property = NULL;
        size_t propertySize = 0;

        if (data == NULL) {
                ALOGE("%s line: %d data is NULL!", __FUNCTION__, __LINE__);
        } else if (dataSize < sizeof(int)) {
                ALOGE("%s line: %d invalid data size!", __FUNCTION__, __LINE__);
        } else {
                int handle = *(reinterpret_cast<int *>(data));

                session = mSessionList.getSession(fd);
                if (session == NULL) {
                        ALOGE("%s line: %d session has not been created!", __FUNCTION__, __LINE__);
                        goto error_get_property;
                }

                sensor = mSensorList.getSensor(handle);
                if (sensor == NULL) {
                        ALOGE("%s line: %d sensor: %d does not exists!", __FUNCTION__, __LINE__, handle);
                        goto error_get_property;
                }

                propertySize = sensor->getProperty(session, data + sizeof(int), dataSize - sizeof(int), reinterpret_cast<void **>(&property));
                if (propertySize <= 0) {
                        ALOGE("%s line: %d sensor: %d get property error!", __FUNCTION__, __LINE__, handle);
                        delete [] property;
                        property = NULL;
                }
        }

error_get_property:
        sendMessageAcknowledge(fd, dataSize, propertySize, property);
        if (property != NULL)
                delete [] property;
}

static void handleGetSensorsList(int fd, int dataSize, uint8_t * data)
{
        if (data == NULL) {
                ALOGE("%s line: %d data is NULL!", __FUNCTION__, __LINE__);
                sendMessageAcknowledge(fd, dataSize, 0, NULL);
        } else if (dataSize < sizeof(sensor_list_flag_t)) {
                ALOGE("%s line: %d invalid data size!", __FUNCTION__, __LINE__);
                sendMessageAcknowledge(fd, dataSize, 0, NULL);
        } else {
                sensor_list_flag_t flag = *(reinterpret_cast<sensor_list_flag_t *>(data));
                sensor_list_header_t header;
                const struct sensor_device_t * list;

                header.string_buffer_size = mSensorList.getStringBufferSize();
                header.string_buffer_base = mSensorList.getStringBufferBase();
                if (flag == FLAG_AWARE_LIST) {
                        header.list_size = mSensorList.getAwareList(&list) * sizeof(struct sensor_device_t);
                } else {
                        header.list_size = mSensorList.getSensorList(&list) * sizeof(struct sensor_device_t);
                }
                sensorhub_message_header_t msgHeader;

                msgHeader.require_code = REQUIRE_NONE;
                msgHeader.ack = 1;
                msgHeader.acknowledge_number = dataSize;
                msgHeader.data_size = sizeof(sensor_list_header_t) + header.list_size + header.string_buffer_size;
                if (send(fd, reinterpret_cast<void *>(&msgHeader), sizeof(msgHeader), 0) < 0) {
                        ALOGE("%s line: %d: send message to client error: %s", __FUNCTION__, __LINE__, strerror(errno));
                        return;
                }

                if (send(fd, reinterpret_cast<void *>(&header), sizeof(header), 0) < 0) {
                        ALOGE("%s line: %d: send message to client error: %s", __FUNCTION__, __LINE__, strerror(errno));
                        return;
                }

                if (send(fd, reinterpret_cast<const void *>(list), header.list_size, 0) < 0) {
                        ALOGE("%s line: %d: send message to client error: %s", __FUNCTION__, __LINE__, strerror(errno));
                        return;
                }

                if (send(fd, reinterpret_cast<void *>(header.string_buffer_base), header.string_buffer_size, 0) < 0) {
                        ALOGE("%s line: %d: send message to client error: %s", __FUNCTION__, __LINE__, strerror(errno));
                        return;
                }
        }
}

static void handleMessage(int fd, sensorhub_message_header_t &messageHeader, int dataSize, uint8_t * data)
{
        if (dataSize < messageHeader.data_size) {
                ALOGE("%s line: %d recieve data incomplete! Expect: %d Actual: %d",
                      __FUNCTION__, __LINE__, messageHeader.data_size, dataSize);
                sendMessageAcknowledge(fd, dataSize, 0, NULL);
                return;
        }

        switch (messageHeader.require_code) {
        case REQUIRE_NONE:
                sendMessageAcknowledge(fd, dataSize, 0, NULL);
                break;
        case REQUIRE_CREATE_SESSION_DATA_FD:
                handleCreateSessionDataFd(fd);
                break;
        case REQUIRE_CREATE_SESSION_CTRL_FD:
                handleCreateSessionCtrlFd(fd, dataSize, data);
                break;
        case REQUIRE_START_STREAMING:
                handleStartStreaming(fd, dataSize, data);
                break;
        case REQUIRE_STOP_STREAMING:
                handleStopStreaming(fd, dataSize, data);
                break;
        case REQUIRE_FLUSH_STREAMING:
                handleFlushStreaming(fd, dataSize, data);
                break;
        case REQUIRE_SET_PROPERTY:
                handleSetProperty(fd, dataSize, data);
                break;
        case REQUIRE_GET_PROPERTY:
                handleGetProperty(fd, dataSize, data);
                break;
        case REQUIRE_GET_SENSORS_LIST:
                handleGetSensorsList(fd, dataSize, data);
                break;
        default:
                break;
        }
}

int main(int argc, char * argv[])
{
        int sockFd, maxFd;
        struct sockaddr_un server;
        fd_set listenFds, readFds;
        pthread_t tid;
        int err;
        struct stat sock_buf;

        fflush(stdout);
        setbuf(stdout, NULL);

        err = pthread_create(&tid, NULL, sensorEventsListener, NULL);
        if (err) {
                ALOGE("%s line: %d Thread create error: %s", __FUNCTION__, __LINE__, strerror(err));
                return -1;
        }

//        sockFd = android_get_control_socket(SENSORHUBD_SOCKET_PATH);
        sockFd = socket(AF_UNIX, SOCK_STREAM, 0);
        if (sockFd < 0) {
                ALOGE("%s line: %d create socket error: %s", __FUNCTION__, __LINE__, strerror(errno));
                return -1;
        }
        ALOGD("%s line: %d sockFd: %d", __FUNCTION__, __LINE__, sockFd);
        server.sun_family = AF_UNIX;
        strcpy(server.sun_path, SENSORHUBD_SOCKET_PATH);
        if (stat(SENSORHUBD_SOCKET_PATH, &sock_buf) == 0) {
                ALOGD_IF(DEBUG, "%s line: %d socket exists, unlink it", __FUNCTION__, __LINE__);
                if (unlink(SENSORHUBD_SOCKET_PATH)) {
                        ALOGE("%s line: %d unlink error: %s", __FUNCTION__, __LINE__, strerror(errno));
                        return -1;
                }
        }

        if (bind(sockFd, (struct sockaddr *) &server, sizeof(struct sockaddr_un)) < 0) {
                ALOGE("%s line: %d bind error: %s", __FUNCTION__, __LINE__, strerror(errno));
                return -1;
        }
                
        if (listen(sockFd, MAX_Q_LENGTH) < 0) {
                ALOGE("%s line: %d listen error: %s", __FUNCTION__, __LINE__, strerror(errno));
                return -1;
        }
                
        FD_ZERO(&listenFds);
        FD_SET(sockFd, &listenFds);
        maxFd = sockFd;

        std::map<int, PlatformService *> platformServiceTable;
        for (int i = 0 ; i < serviceList.size(); i++) {
                PlatformService *service = serviceList[i];
                int notifyFd = service->getNotifyFd();
                if (notifyFd > 0) {
                        platformServiceTable[notifyFd] = service;
                        FD_SET(notifyFd, &listenFds);
                        if (notifyFd > maxFd)
                                maxFd = notifyFd;
                }
        }

        while (true) {
                readFds = listenFds;

                err = select(maxFd + 1, &readFds, NULL, NULL, NULL);
                if (err < 0) {
                        if (errno != EINTR)
                                ALOGE("%s line: %d select error: %s", __FUNCTION__, __LINE__, strerror(errno));
                        continue;
                }

                std::map<int, PlatformService *>::iterator it;
                for (it = platformServiceTable.begin(); it != platformServiceTable.end(); it++) {
                        int notifyFd = it->first;
                        PlatformService *service = it->second;
                        if (FD_ISSET(notifyFd, &readFds)) {
                                int notification;
                                read(notifyFd, &notification, sizeof(int));
                                service->handleNotify(notification);
                                FD_CLR(notifyFd, &readFds);
                        }
                }

                /* handle new connection request */
                if (FD_ISSET(sockFd, &readFds)) {
                        struct sockaddr_un client_addr;
                        socklen_t addrlen = sizeof(client_addr);
                        int clientFd = accept(sockFd, reinterpret_cast<struct sockaddr *>(&client_addr), &addrlen);
                        if (clientFd == -1) {
                                ALOGE("%s line: %d accept error: %s", __FUNCTION__, __LINE__, strerror(errno));
                        } else {
                                FD_SET(clientFd, &listenFds);
                                if (clientFd > maxFd)
                                        maxFd = clientFd;
                        }
                        FD_CLR(sockFd, &readFds);
                }

                for (int fd = maxFd; fd >= 0; fd--) {
                        sensorhub_message_header_t messageHeader;
                        uint8_t * data = NULL;

                        if (!FD_ISSET(fd, &readFds))
                                continue;

                        int length = recv(fd, reinterpret_cast<void *>(&messageHeader), sizeof(messageHeader), 0);
                        if (length == sizeof(messageHeader)) {
                                if (messageHeader.data_size > 0) {
                                        data = new uint8_t[messageHeader.data_size];
                                        length = recv(fd, reinterpret_cast<void *>(data), messageHeader.data_size, 0);
                                }
                        } else if (length > 0) {
                                ALOGE("%s line: %d recv invalid header length: %d", __FUNCTION__, __LINE__, length);
                                sendMessageAcknowledge(fd, 0, 0, NULL);
                                continue;
                        }
                        if (length <= 0) {
                                ALOGW("%s line: %d remove session(fd %d): %s", __FUNCTION__, __LINE__, fd, length == 0 ? "no message" : strerror(errno));
                                mSessionList.removeSession(fd);
                                // close fds here for fds are managed by this module and may not related to a session
                                close(fd);
                                FD_CLR(fd, &listenFds);
                        } else {
                                handleMessage(fd, messageHeader, length, data);
                        }
                        if (data != NULL)
                                delete [] data;
                }

        }

        pthread_join(tid, NULL);

        return 0;
}
