/*
 * Copyright (C) 2008-2009 Patrick Ohly <patrick.ohly@gmx.de>
 * Copyright (C) 2009 Intel Corporation
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) version 3.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301  USA
 */

#include "config.h"
#include <syncevo/util.h>
#include <syncevo/SyncContext.h>
#include <syncevo/TransportAgent.h>
#include <syncevo/Logging.h>

#include <synthesis/syerror.h>

#include <boost/scoped_array.hpp>
#include <boost/foreach.hpp>
#include <fstream>

#include <errno.h>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <dirent.h>

#ifdef ENABLE_UNIT_TESTS
#include "test.h"
CPPUNIT_REGISTRY_ADD_TO_DEFAULT("SyncEvolution");
#endif

#include <syncevo/declarations.h>
SE_BEGIN_CXX

string normalizePath(const string &path)
{
    string res;

    res.reserve(path.size());
    size_t index = 0;
    while (index < path.size()) {
        char curr = path[index];
        res += curr;
        index++;
        if (curr == '/') {
            while (index < path.size() &&
                   (path[index] == '/' ||
                    (path[index] == '.' &&
                     index + 1 < path.size() &&
                     (path[index + 1] == '.' ||
                      path[index + 1] == '/')))) {
                index++;
            }
        }
    }
    if (!res.empty() && res[res.size() - 1] == '/') {
        res.resize(res.size() - 1);
    }
    return res;
}

void mkdir_p(const string &path)
{
    boost::scoped_array<char> dirs(new char[path.size() + 1]);
    char *curr = dirs.get();
    strcpy(curr, path.c_str());
    do {
        char *nextdir = strchr(curr, '/');
        if (nextdir) {
            *nextdir = 0;
            nextdir++;
        }
        if (*curr) {
            if (access(dirs.get(),
                       nextdir ? (R_OK|X_OK) : (R_OK|X_OK|W_OK)) &&
                (errno != ENOENT ||
                 mkdir(dirs.get(), 0700))) {
                SyncContext::throwError(string(dirs.get()), errno);
            }
        }
        if (nextdir) {
            nextdir[-1] = '/';
        }
        curr = nextdir;
    } while (curr);
}

void rm_r(const string &path, boost::function<bool (const string &,
                                                    bool)> filter)
{
    struct stat buffer;
    if (lstat(path.c_str(), &buffer)) {
        if (errno == ENOENT) {
            return;
        } else {
            SyncContext::throwError(path, errno);
        }
    }

    if (!S_ISDIR(buffer.st_mode)) {
        if (!filter(path, false) ||
            !unlink(path.c_str())) {
            return;
        } else {
            SyncContext::throwError(path, errno);
        }
    }

    ReadDir dir(path);
    BOOST_FOREACH(const string &entry, dir) {
        rm_r(path + "/" + entry, filter);
    }
    if (filter(path, true) &&
        rmdir(path.c_str())) {
        SyncContext::throwError(path, errno);
    }
}

bool isDir(const string &path)
{
    DIR *dir = opendir(path.c_str());
    if (dir) {
        closedir(dir);
        return true;
    } else if (errno != ENOTDIR && errno != ENOENT) {
        SyncContext::throwError(path, errno);
    }

    return false;
}

UUID::UUID()
{
    static class InitSRand {
    public:
        InitSRand() {
            ifstream seedsource("/dev/urandom");
            unsigned int seed;
            if (!seedsource.get((char *)&seed, sizeof(seed))) {
                seed = time(NULL);
            }
            srand(seed);
        }
    } initSRand;

    char buffer[16 * 4 + 5];
    sprintf(buffer, "%08x-%04x-%04x-%02x%02x-%08x%04x",
            rand() & 0xFFFFFFFF,
            rand() & 0xFFFF,
            (rand() & 0x0FFF) | 0x4000 /* RFC 4122 time_hi_and_version */,
            (rand() & 0xBF) | 0x80 /* clock_seq_hi_and_reserved */,
            rand() & 0xFF,
            rand() & 0xFFFFFFFF,
            rand() & 0xFFFF
            );
    this->assign(buffer);
}


ReadDir::ReadDir(const string &path, bool throwError) : m_path(path)
{
    DIR *dir = NULL;

    try {
        dir = opendir(path.c_str());
        if (!dir) {
            SyncContext::throwError(path, errno);
        }
        errno = 0;
        struct dirent *entry = readdir(dir);
        while (entry) {
            if (strcmp(entry->d_name, ".") &&
                strcmp(entry->d_name, "..")) {
                m_entries.push_back(entry->d_name);
            }
            entry = readdir(dir);
        }
        if (errno) {
            SyncContext::throwError(path, errno);
        }
    } catch(...) {
        if (dir) {
            closedir(dir);
        }
        if (throwError) {
            throw;
        } else {
            return;
        }
    }

    closedir(dir);
}

std::string ReadDir::find(const string &entry, bool caseSensitive)
{
    BOOST_FOREACH(const string &e, *this) {
        if (caseSensitive ? e == entry : boost::iequals(e, entry)) {
            return m_path + "/" + e;
        }
    }
    return "";
}

bool ReadFile(const string &filename, string &content)
{
    ifstream in;
    in.open(filename.c_str());
    ostringstream out;
    char buf[8192];
    do {
        in.read(buf, sizeof(buf));
        out.write(buf, in.gcount());
    } while(in);

    content = out.str();
    return in.eof();
}

unsigned long Hash(const char *str)
{
    unsigned long hashval = 5381;
    int c;

    while ((c = *str++) != 0) {
        hashval = ((hashval << 5) + hashval) + c;
    }

    return hashval;
}

std::string StringPrintf(const char *format, ...)
{
    va_list ap;
    va_start(ap, format);
    std::string res = StringPrintfV(format, ap);
    va_end(ap);
    return res;
}

std::string StringPrintfV(const char *format, va_list ap)
{
    va_list aq;

    char *buffer = NULL, *nbuffer = NULL;
    ssize_t size = 0;
    ssize_t realsize = 255;
    do {
        // vsnprintf() destroys ap, so make a copy first
        va_copy(aq, ap);

        if (size < realsize) {
            nbuffer = (char *)realloc(buffer, realsize + 1);
            if (!nbuffer) {
                if (buffer) {
                    free(buffer);
                }
                return "";
            }
            size = realsize;
            buffer = nbuffer;
        }

        realsize = vsnprintf(buffer, size + 1, format, aq);
        if (realsize == -1) {
            // old-style vnsprintf: exact len unknown, try again with doubled size
            realsize = size * 2;
        }
        va_end(aq);
    } while(realsize > size);

    std::string res = buffer;
    free(buffer);
    return res;
}

SyncMLStatus Exception::handle(SyncMLStatus *status, Logger *logger)
{
    SyncMLStatus new_status = STATUS_FATAL;

    try {
        throw;
    } catch (const TransportException &ex) {
        SE_LOG_DEBUG(logger, NULL, "TransportException thrown at %s:%d",
                     ex.m_file.c_str(), ex.m_line);
        SE_LOG_ERROR(logger, NULL, "%s", ex.what());
        new_status = SyncMLStatus(sysync::LOCERR_TRANSPFAIL);
    } catch (const Exception &ex) {
        SE_LOG_DEBUG(logger, NULL, "exception thrown at %s:%d",
                     ex.m_file.c_str(), ex.m_line);
        SE_LOG_ERROR(logger, NULL, "%s", ex.what());
    } catch (const std::exception &ex) {
        SE_LOG_ERROR(logger, NULL, "%s", ex.what());
    } catch (...) {
        SE_LOG_ERROR(logger, NULL, "unknown error");
    }

    if (status && *status == STATUS_OK) {
        *status = new_status;
    }
    return status ? *status : new_status;
}

std::string SubstEnvironment(const std::string &str)
{
    std::stringstream res;
    size_t envstart = std::string::npos;
    size_t off;

    for(off = 0; off < str.size(); off++) {
        if (envstart != std::string::npos) {
            if (str[off] == '}') {
                std::string envname = str.substr(envstart, off - envstart);
                envstart = std::string::npos;

                const char *val = getenv(envname.c_str());
                if (val) {
                    res << val;
                } else if (envname == "XDG_CONFIG_HOME") {
                    res << getHome() << "/.config";
                } else if (envname == "XDG_DATA_HOME") {
                    res << getHome() << "/.local/share";
                } else if (envname == "XDG_CACHE_HOME") {
                    res << getHome() << "/.cache";
                }
            }
        } else {
            if (str[off] == '$' &&
                off + 1 < str.size() &&
                str[off + 1] == '{') {
                envstart = off + 2;
                off++;
            } else {
                res << str[off];
            }
        }
    }

    return res.str();
}

SE_END_CXX
